package ru.yandex.autotests.direct.scriptrunner.client;

import java.io.Closeable;
import java.util.List;

import com.google.gson.Gson;
import org.apache.commons.io.IOUtils;
import org.apache.http.HttpEntity;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;

import ru.yandex.autotests.direct.scriptrunner.service.clientdata.ClientDataParams;
import ru.yandex.autotests.direct.scriptrunner.service.exportmaster.ExportMasterParams;
import ru.yandex.autotests.direct.scriptrunner.service.scripts.ScriptDBConfig;
import ru.yandex.autotests.direct.scriptrunner.service.scripts.ScriptParams;
import ru.yandex.autotests.direct.scriptrunner.service.scripts.ScriptRunParams;
import ru.yandex.autotests.direct.scriptrunner.service.scripts.ScriptRunResult;
import ru.yandex.autotests.direct.scriptrunner.service.scripts.ScriptRunTask;
import ru.yandex.autotests.direct.scriptrunner.service.scripts.ScriptrunnerClientException;
import ru.yandex.autotests.direct.scriptrunner.service.scripts.ScriptrunnerServerException;
import ru.yandex.autotests.direct.scriptrunner.service.scripts.Scripts;
import ru.yandex.autotests.directapi.darkside.connection.DarksideConfig;
import ru.yandex.autotests.directapi.darkside.connection.IntapiConfig;

@SuppressWarnings("unused")
public class ScriptRunnerClient {

    private static final int TIMEOUT = 180000;

    private DarksideConfig darksideConfig;
    private ScriptRunnerClientConfig scriptrunnerConfig = new ScriptRunnerClientConfig();

    public ScriptRunnerClient(DarksideConfig darksideConfig) {
        this.darksideConfig = darksideConfig;
    }

    public ScriptRunResult runBsExportMaster(int cid, String dbHost, int shardId) {

        return runScript(Scripts.BS_EXPORT_MASTER
                , new ExportMasterParams()
                        .withCid(cid)
                        .withShardId(shardId)
                        .withNoResync(true)
                        .withOnce(true)
                , dbHost, shardId);
    }

    public ScriptRunResult runBsExportMaster(List<Integer> cidList, String dbHost, int shardId) {

        return runScript(Scripts.BS_EXPORT_MASTER
                , new ExportMasterParams()
                        .withCid(cidList)
                        .withShardId(shardId)
                        .withNoResync(true)
                        .withOnce(true)
                , dbHost, shardId);
    }

    public ScriptRunResult runBsClientData(ClientDataParams params, String dbHost) {
        return runScript(Scripts.BS_CLIENT_DATA, params, dbHost, params.getShardId());
    }

    private ScriptRunResult runScript(String scriptName, ScriptParams scriptParams, String dbUrl, int shardId) {
        ScriptRunTask task = buildTask(scriptParams, dbUrl, shardId);
        String requestUrl = buildRequestUrl(scriptName);
        return executeScriptRunTask(task, requestUrl);
    }

    private ScriptRunResult executeScriptRunTask(ScriptRunTask scriptRunTask, String url) {
        CloseableHttpClient client = HttpClients.createDefault();

        HttpPost request = new HttpPost(url);
        RequestConfig requestConfig = RequestConfig.custom()
                .setSocketTimeout(TIMEOUT)
                .setConnectTimeout(TIMEOUT)
                .setConnectionRequestTimeout(TIMEOUT)
                .build();
        HttpEntity entity = new StringEntity(new Gson().toJson(scriptRunTask), ContentType.APPLICATION_JSON);
        request.setConfig(requestConfig);
        request.setEntity(entity);

        try {
            CloseableHttpResponse response = client.execute(request);
            String content = IOUtils.toString(response.getEntity().getContent());
            if (response.getStatusLine().getStatusCode() >= 500) {
                throw new ScriptrunnerServerException("серверная ошибка при выполнении скрипта:\n" + content);
            }
            try {
                return new Gson().fromJson(content, ScriptRunResult.class);
            } finally {
                closeQuietly(response);
            }
        } catch (Exception e) {
            throw new ScriptrunnerClientException("ошибка при выполнении скрипта через Scriptrunner", e);
        } finally {
            closeQuietly(client);
        }
    }

    private ScriptRunTask buildTask(ScriptParams scriptParams, String dbUrl, int shardId) {
        String betaHost = ((IntapiConfig) darksideConfig).getIPV6Host();
        ScriptRunParams scriptRunParams = new ScriptRunParams(betaHost, scriptParams);
        return new ScriptRunTask(new ScriptDBConfig(dbUrl, shardId), scriptRunParams);
    }

    private String buildRequestUrl(String scriptName) {
        return scriptrunnerConfig.getServiceUrl() + "/scripts/" + scriptName;
    }

    private void closeQuietly(CloseableHttpClient client) {
        try {
            client.close();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void closeQuietly(Closeable closeable) {
        try {
            closeable.close();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}
