package ru.yandex.autotests.direct.scriptrunner.service.scripts.clients;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.type.TypeFactory;
import com.googlecode.jsonrpc4j.JsonRpcHttpClient;
import com.googlecode.jsonrpc4j.ReflectionUtil;
import org.apache.http.client.utils.URIBuilder;

import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.net.*;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;

/**
 * User: omaz [origin at direct-darkside], mexicano
 * Date: 26.08.13
 * Json-rpc клиент
 */
public class IntapiJsonRpcClient {

    private static final int READ_TIMEOUT = 60 * 1000;
    private static final int CONNECT_TIMEOUT = 20 * 1000;
    private static final int MAX_TRIES = 3;

    private URI jsonRpcUrl;
    private int connectTimeout;
    private final ObjectMapper objectMapper;


    public IntapiJsonRpcClient(URI jsonRpcUrl, int connectTimeout) {
        this.jsonRpcUrl = jsonRpcUrl;
        this.connectTimeout = connectTimeout > 0 ? connectTimeout : CONNECT_TIMEOUT;
        this.objectMapper = new ObjectMapper()
                .configure(SerializationFeature.INDENT_OUTPUT, true)
                .setSerializationInclusion(JsonInclude.Include.NON_NULL);
    }

    public <T> T getServiceWithNamedParams(String serviceName, Class<T> tClass) {
        JsonRpcHttpClient jsonRpcHttpClient = initClient(serviceName);
        return createCustomClientProxy(tClass, jsonRpcHttpClient, true);
    }

    private JsonRpcHttpClient initClient(String serviceName) {
        URL url;
        try {
            URIBuilder builder = new URIBuilder(jsonRpcUrl);
            builder.setPath(builder.getPath() + "/" + serviceName);
            url = builder.build().toURL();
        } catch (URISyntaxException | MalformedURLException e) {
            throw new IllegalArgumentException("Error in JsonRPC service URL...", e);
        }

        JsonRpcHttpClient jsonRpcHttpClient = new JsonRpcHttpClient(
                objectMapper, url, new HashMap<String, String>());
        disableSslVerification(jsonRpcHttpClient);
        jsonRpcHttpClient.setReadTimeoutMillis(READ_TIMEOUT);
        jsonRpcHttpClient.setConnectionTimeoutMillis(connectTimeout);
        return jsonRpcHttpClient;
    }

    @SuppressWarnings("unchecked")
    // create and return the proxy
    private <T> T createCustomClientProxy(Class<T> proxyInterface,
                                          final JsonRpcHttpClient client,
                                          final boolean useNamedParams) {
        return (T) Proxy.newProxyInstance(getClass().getClassLoader(), new Class<?>[]{proxyInterface},
                new InvocationHandler() {
                    public Object invoke(Object proxy, Method method, Object[] args) throws Exception {
                        IntapiMessage message = new IntapiMessage();

                        // Получаем ответ в Object, логируем ответ а потом пытаемся распарсить.
                        // В случае исключения SocketTimeoutException повторяем заданное количество попыток
                        Object response = null;

                        for (int i = 1; i <= MAX_TRIES; i++) {
                            try {
                                message.setUrl(client.getServiceUrl().toString() + "?method=" + method.getName());
                                Object argument = useNamedParams ?
                                        ReflectionUtil.parseArguments(method, args, true) : args[0];
                                message.setRequest(objectMapper.writeValueAsString(argument));

                                response = client.invoke(
                                        method.getName(), argument, Object.class, new HashMap<String, String>());
                                message.setResponse(objectMapper.writeValueAsString(response));
                                break;
                            } catch (SocketTimeoutException e) {
                                if (i >= MAX_TRIES) {
                                    message.setResponse(e.toString());
                                    String errorText = String.format(
                                            "Error in call of method \"%s\" (try: %s)\nError: %s",
                                            method.getName(), i, e.toString());
                                    throw new IllegalStateException(errorText, e);
                                }
                            } catch (Throwable e) {
                                message.setResponse(e.toString());
                                String errorText = String.format(
                                        "Error in call of method \"%s\" (try: %s)\nError: %s",
                                        method.getName(), i, e.toString());
                                throw new IllegalStateException(errorText, e);
                            }
                        }

                        try {
                            JavaType returnJavaType =
                                    TypeFactory.defaultInstance().constructType(method.getGenericReturnType());
                            return objectMapper.convertValue(response, returnJavaType);
                        } catch (Exception e) {
                            String errorText =
                                    String.format("Can not parse response: %s\n" +
                                                    "Expected response type: %s\n" +
                                                    "Response: " + objectMapper.writeValueAsString(response),
                                            method.getName(), method.getReturnType());
                            throw new IllegalStateException(errorText, e);
                        }
                    }
                });
    }

    private void disableSslVerification(JsonRpcHttpClient jsonRpcHttpClient) {
        try {
            // Create a trust manager that does not validate certificate chains
            TrustManager[] trustAllCerts = new TrustManager[]{new CertificatesUtils.IgnoreSSLTrustManager()};

            // Install the all-trusting trust manager
            SSLContext sc = SSLContext.getInstance("SSL");
            sc.init(null, trustAllCerts, new java.security.SecureRandom());
            jsonRpcHttpClient.setSslContext(sc);

            // Install the all-trusting host verifier
            jsonRpcHttpClient.setHostNameVerifier(new CertificatesUtils.AllowAllHostnameVerifier());
        } catch (NoSuchAlgorithmException | KeyManagementException e) {
            throw new IllegalStateException("Can not turn off SslVerification for JsonRpcHttpClient", e);
        }
    }
}
