#!/usr/bin/python
# -*- coding: utf8 -*-

description = """
Скрипт берет описание объектов из Логброкера и записывает в файлы.
Формат файлов -- наш собственный, общий со скриптом create-topics.py

Скрипт в разработке, может делать неожиданное.

Для регулярного запуска пока не предназначен (умеет только полный дамп, diff не умеет).
Пока нужен только для того, чтобы получить схемы существующих объектов, чтобы было что давать соседнему скрипту create-topic.py

Пока не соглашается работать с существующим каталогом, хочет все создать сам (потому что делает дамп, не дифф).

После того, как будет уметь diff, стоит доделать аркадийную сборку, регулярный запуск и мониторинг соответствия файлов и LogBroker.

Пример использования:
задампить аккаунт direct из кластера lbkx, записать в каталог lbkx/direct:
./dump-topics.py -s lbkx -a /direct -d lbkx/direct
"""


import os
import re
import sys

sys.path.insert(0, '/opt/direct-py/startrek-python-client-sni-fix')

import argparse
import copy
import getpass
import json
import requests
import socket
import subprocess
import time
import tempfile

prop_name_translations = {
        "partitions count": "partitions",
        "retention period": "retention",
        "important": "important",
        "ABC service": "abc_service",
        "limits mode": "limits_mode",
        "responsible": "responsible",
        }
prop_value_translations = {
        "<empty>": "",
        "False": 0,
        "True": 1,
        }

def die(message=''):
    # комментарий от andy-ilyin@: может, лучше raise по месту с приблизительно подходящим текстом? стектрейс терять не хочется
    sys.stderr.write("%s\n" % message)
    exit(1)


def my_qx(cmd, verbose=False):
   if verbose:
       print("going to exec: %s\n" % cmd)

   res = subprocess.check_output(cmd)
   try:
       res = subprocess.check_output(cmd)
   except:
       die("cmd failed, stop (%s)" % cmd)

   return res


def parse_options():
    my_name = os.path.basename( sys.argv[0] )
    parser = argparse.ArgumentParser(add_help=False)
    parser.add_argument("-h", "--help", dest="help", help="Справка", action="store_true")
    parser.add_argument("-s", "--server", dest="logbroker_server", help="Сервер logbroker, передается в logbroker -s", type=str)
    parser.add_argument("-a", "--account", dest="logbroker_account", help="Аккаунт в  logbroker", type=str)
    parser.add_argument("-d", "--directory", dest="schema_directory", help="Корневой каталог логброкерных схем", type=str)
    opts, extra = parser.parse_known_args()

    if opts.help:
        print description
        print parser.format_help()
        exit(0)

    opts.extra = extra

    return opts


def get_line_range(lines, start_regexp, stop_regexp):
    (started, stopped) = (False, False)
    res = []

    for l in lines:
        if not started and re.match(start_regexp, l):
            started = True
            stopeed = False
        if started and not stopped:
            res.append(l)
        if  started and not stopped and re.match(stop_regexp, l):
            stopped = True
            started = False

    return res


def table_to_arrs(lines):
    res = []
    for l in lines:
        values = [ v.strip() for v in re.split(r' +\| +', l)]
        res.append(values)
    return res


def get_object_description(logbroker_server, topic_path):
    desc = {
            "permissions": {},
            "read_rules": [],
            "properties": {},
            "logbroker_server": logbroker_server,
            "path": topic_path,
            "type": None,
            "owner": None,
            }
    desc_text = my_qx(["logbroker", "-s", logbroker_server, "schema", "describe", topic_path])

    # компенсация наивного разбора таблиц
    # значения в таблицах разбираем по нескольким пробелам
    # в одном названии проперти есть два пробела, убираем
    desc_text = re.sub(r'allow unauthenticated  read via legacy HTTP API', 'allow unauthenticated read via legacy HTTP API', desc_text)

    lines = desc_text.split("\n")

    # тип объекта
    m = re.match(r'^Type:\s+(\S+)\s*$', lines[0])
    if not m:
        die("can't determine type of object '%s/%s'" % (logbroker_server, topic_path))
    desc['type'] = m.group(1).lower()

    # владелец
    owner = re.match(r'^Owner:\s+(\S+)\s*$', lines[5])
    if owner:
        desc['owner'] = owner.group(1)

    # Permissions
    perm_lines = get_line_range(lines, r'^Permissions$', r'^$')
    perm_lines = [ p for p in perm_lines if not re.match(r'^( *|Permissions|-+\+-+|\s+subject\s+\|\s+permissions\s*)$', p) ]
    rows = table_to_arrs(perm_lines)
    for r in rows:
        if len(r) != 2:
            die("can't parse permission '%s', topic %s" % (" ".join(r), topic_path))
        perms = [ p for p in r[1].split(" ") if p != "" ]
        desc['permissions'][ r[0] ] = perms

    # Read rules
    read_rules_lines = get_line_range(lines, r'^Read rules$', r'^$')
    read_rules_lines = [ p for p in read_rules_lines if not re.match(r'^( *|Read rules|-+\+-+|\s*consumer\s+\|\s+mode\s*|\s*topic\s+\|\s+mode\s*)$', p) ]
    rows = table_to_arrs(read_rules_lines)
    for r in rows:
        if len(r) != 2:
            die("can't parse read rule '%s', topic %s" % (" ".join(r), topic_path))
        mode = r[1]
        desc['read_rules'].append([ r[0], mode])

    # Properties
    props_lines = get_line_range(lines, r'^Properties$', r'^$')
    props_lines = [ p for p in props_lines if not re.match(r'^( *|Properties|-+\+-+\+-+|\s*name\s+\|\s+value\s+\|\s+source\s*)$', p) ]
    rows = table_to_arrs(props_lines)
    for r in rows:
        if len(r) != 3:
            die("can't parse property '%s', topic %s" % ("  ".join(r), topic_path))
        if r[0] in prop_name_translations:
            value = r[1]
            if value in prop_value_translations:
                value = prop_value_translations[value]
            desc['properties'][ prop_name_translations[r[0]] ] = value

    return desc


def dump_logbroker_objects( logbroker_server, logbroker_path, fs_path ):
    """
    рекурсивно обходим все каталоги и объекты в логброкере, создаем соответствующие каталоги и файлы на файловой системе
    logbroker_server -- сервер логброкера
    logbroker_path -- текущий обрабатываемый логброкерный путь
    fs_path -- текущий каталог в файловой системе
    """
    # Берем список всего, что есть в логброкере на текущем уровне
    if not os.path.exists(fs_path):
        os.mkdir(fs_path)
    text = my_qx(["logbroker", "-s", logbroker_server, "schema", "list", logbroker_path])
    lines = text.split("\n")

    objects = []
    for l in lines:
        if re.match(r'^total:\s+[0-9]+', l):
            continue
        if l == '':
            continue
        m = re.match(r'^<([^ ]+)>\s+([^ ]+)', l)
        if not m:
            die("can't parse line '%s', current path %s" % (l, logbroker_path))
        o = {
                "type": m.group(1),
                "name": m.group(2),
                }
        objects.append(o)

    # 1. обходим все объекты и пишем файлы
    for o in objects:
        if o["type"] == "directory":
            continue
        print "found %s %s (%s)" % ( o["type"], o["name"], logbroker_path )
        if o["type"] == "topic" or o["type"] == "consumer":
            filename = "%s/%s.%s.json" % (fs_path, o["name"], o["type"])
            desc = get_object_description(logbroker_server, "%s/%s" % (logbroker_path, o["name"]))
            with open(filename, 'w') as outfile:
                json.dump(desc, outfile, indent=4, separators=(",", ": "), sort_keys=True)
        else:
            print "UNKNOWN type %s" % o["type"]


    # 2. обходим все директории и рекурсивно обрабатываем их
    for o in objects:
        if o["type"] != "directory":
            continue
        print "going to %s %s (%s)" % ( o["type"], o["name"], logbroker_path )
        new_logbroker_path = "%s/%s" % (logbroker_path, o["name"])
        new_fs_path = "%s/%s" % (fs_path, o["name"])
        dump_logbroker_objects(logbroker_server, new_logbroker_path, new_fs_path)

    return


def run():
    opts = parse_options()

    dump_logbroker_objects( opts.logbroker_server, opts.logbroker_account, opts.schema_directory )

    exit(0)



if __name__ == '__main__':
    run();

