from juggler_sdk import Check, Child

from library.python.monitoring.solo.objects.solomon.v2 import Alert, Type, Expression
from library.python.monitoring.solo.objects.solomon.sensor import Sensor
from direct.solo.registered.juggler.reference import CALLING_CHECKS_CHAT
from direct.solo.registered.juggler.reference import DUTY_ESCALATION
from direct.solo.registered.channel import channels
from direct.solo.registered.project import projects

alert_configs = {
    'ess-moderation-routing-lag': {
        'topic': 'modadvert/modadvert-sm-direct-general-results-log',
        'consumer': 'direct/moderation/routing-consumer',
        'account': 'modadvert',
        'description': 'Отставание обработки топика с вердиктами джобой ReceiveModerationResponseRouterJob',
        'threshold_minutes': 30
    },
    'ess-moderation-apply-general-results-lag': {
        'topic': 'direct/moderation/general-results',
        'consumer': 'direct/modadvert/general/requests-consumer',
        'account': 'direct',
        'description': 'Отставание обработки топика с вердиктами general-results',
        'threshold_minutes': 30
    },
    'ess-moderation-apply-image-ad-results-lag': {
        'topic': 'direct/moderation/image_ad_sm-results',
        'consumer': 'direct/modadvert/general/requests-consumer',
        'account': 'direct',
        'description': 'Отставание обработки топика с вердиктами image_ad_sm',
        'threshold_minutes': 120
    },
    'ess-moderation-apply-smart-banner-results-lag': {
        'topic': 'direct/moderation/smart_banner-results',
        'consumer': 'direct/modadvert/general/requests-consumer',
        'account': 'direct',
        'description': 'Отставание обработки топика с вердиктами smart_banner',
        'threshold_minutes': 60
    },
    'ess-moderation-apply-text-banner-results-lag': {
        'topic': 'direct/moderation/text_banner-results',
        'consumer': 'direct/modadvert/general/requests-consumer',
        'account': 'direct',
        'description': 'Отставание обработки топика с вердиктами text_banner',
        'threshold_minutes': 120
    },
}


def generate_alert_and_check(alert_id, alert_config):
    alert = Alert(
        id=alert_id,
        project_id=projects.direct.id,
        name=alert_id,
        annotations={
            "host": "direct-lb-moderation",
            "service": alert_config.get('service', alert_id),
            "description": "{{{{#isAlarm}}}}{topic} processing lag by {consumer} is greater than {threshold_minutes} "
                           "minutes{{{{/isAlarm}}}}".format(**alert_config)
        },
        description=alert_config['description'],
        type=Type(
            expression=Expression(
                program="""
    let lag = group_lines('last', {sensor});
    let avg_lag = avg(lag);
    alarm_if(avg_lag > {threshold_millis});
                """.format(
                    sensor=Sensor(
                        project="kikimr", cluster="lbkx", service="pqtabletAggregatedCounters",
                        host="*",
                        sensor="CreateTimeLagMsByCommitted",
                        TopicPath=alert_config['topic'],
                        ConsumerPath=alert_config['consumer'],
                        Account=alert_config['account'],
                        OriginDC="*",
                        partition="-",
                        user_counters="PersQueue",
                    ),
                    threshold_millis=alert_config['threshold_minutes'] * 60000
                ),
            )
        ),
        notification_channels={channels.direct_juggler.id},
        window_secs=10 * 60,
        delay_seconds=0,
    )

    check = Check(
        namespace="direct.prod",
        host=alert.annotations["host"],
        service=alert.annotations["service"],
        refresh_time=90, ttl=900,
        aggregator="logic_or",
        # tags=["is_disaster", "marty", "warden_alert_create_spi", "call_needed", "disaster_alert", "yabs_partner"],
        tags=[
            'directadmin.calls',
            'direct_group_sre',
        ],
        meta={
            "urls": [
                {
                    "type": "wiki",
                    "title": "документация",
                    "url": "https://docs.yandex-team.ru/direct-dev/reference/alerts/ess-moderation-apply-results-lag"
                },
            ]
        },
        children=[Child(
            host=alert.annotations["host"],
            service=alert.annotations["service"],
            group_type="HOST"
        )],
        notifications=[
            CALLING_CHECKS_CHAT,
            DUTY_ESCALATION,
        ],
    )

    return [alert, check]


def get_alerts():
    return [
        alert
        for (id, config) in alert_configs.items()
        for alert in generate_alert_and_check(id, config)
    ]
