import ast
import pkgutil
from itertools import chain

from pkg_resources import resource_listdir

from . import common
from direct.solo.registered.project import projects


def data(project, config):
    monitoring_service = config.get("monitoring_service")
    sensor = config.get("sensor")
    data_str = "project=\"{}\"," \
               "cluster=\"app_java-jobs\"," \
               "service=\"common-metrics\"," \
               "host=\"CLUSTER\"," \
               "sensor=\"{}\"," \
               "monitoring_service=\"{}\"," \
               "bin=\"*\"" \
        .format(project.id, sensor, monitoring_service)
    data_str += common.label_matchers(config)
    return "{" + data_str + "}"


def threshold(configs, default_threshold):
    if configs is None or len(configs) == 0:
        return default_threshold

    if len(configs) == 1:
        return "{} ? {} : {}" \
            .format(common.var_name(configs[0].get("label")),
                    configs[0].get("threshold"),
                    threshold(configs[1:], default_threshold))

    return "{} ? {} : ({})" \
        .format(common.var_name(configs[0].get("label")),
                configs[0].get("threshold"),
                threshold(configs[1:], default_threshold))


def lets(label_values):
    if label_values is None or len(label_values) == 0:
        return ""
    return "\n".join(
        list(map(
            lambda label: "let {} = get_label(data, \"monitoring_method\") == \"{}\";".format(
                common.var_name(label),
                label),
            label_values)))


def program(project, config):
    alert_window_minutes = config.get("alert_window_minutes")
    evaluation_tail_secs = config.get("evaluation_tail_minutes", alert_window_minutes) * 60
    method_configs = config.get("method_configs")
    threshold_avg_rpw = config.get("threshold_avg_rpw", -1)
    if method_configs is None:
        labels = []
    else:
        labels = list(map(lambda conf: conf.get("label"),
                          method_configs))
    return """
        let data = {0};

        let data_rps = histogram_count(data);
        let data_rps_max = avg(data_rps);
        let rpsLine = constant_line(data_rps_max);

        let data = sum(data) by 1m;
        let dataMov = moving_sum(data, {1});
        let result = histogram_percentile({2}, 'bin', dataMov);
        let tail_result = tail(result, {3}s);
        {4}

        let threshold = {5};

        let thresholdLine = constant_line(threshold);
        let max_value = max(tail_result);

        alarm_if(data_rps_max > {6} && max_value > threshold);

""".format(
        data(project, config),
        config.get("moving_sum_interval"),
        config.get("percentile"),
        evaluation_tail_secs,
        lets(labels),
        threshold(config.get("method_configs"), config.get("default_threshold")),
        threshold_avg_rpw)


def get_alerts_internal(project, instance):
    export = []
    folder_path = "config/" + instance + "/threshold/"
    for file in resource_listdir(__package__, folder_path):
        file_path = folder_path + file
        config_str = pkgutil.get_data(__package__, file_path).decode("UTF-8")
        config = ast.literal_eval(config_str)
        juggler_description = "percentile_" + str(config.get("percentile")) + \
                              ": {{expression.avg_current_data}} value {{expression.max_value}} " \
                              "threshold {{expression.threshold}}"
        assert isinstance(config, dict)
        export += common.alert(project, common.alert_name(file), config, juggler_description, program(project, config))
    return export


alerts_prod = get_alerts_internal(projects.direct, "production")
alerts_ts = get_alerts_internal(projects.direct_test, "test")


def get_all_alerts():
    return chain(
        alerts_prod,
        alerts_ts
    )
