from enum import Enum

from library.python.monitoring.solo.objects.solomon.v2 import Alert, Type, Expression
from library.python.monitoring.solo.objects.solomon.sensor import Sensor
from direct.solo.registered.channel import channels
from direct.solo.registered.project import projects
from juggler_sdk import Check, Child

WARN_DISK_BORDER = 80
CRIT_DISK_BORDER = 90


class MdbCluster:
    def __init__(self, db_type, cluster_id, name):
        self.db_type = db_type
        self.cluster_id = cluster_id
        self.name = name


class DbType(Enum):
    MONGO = 1
    REDIS = 2
    MYSQL = 3


DB_TYPE_TO_DISK_USAGE_SENSOR_SUFFIX = {
    DbType.MONGO: 'mongodb',
    DbType.REDIS: 'redis',
    DbType.MYSQL: 'mysql'
}


def create_disk_usage_alert(cluster):
    sensor_suffix = DB_TYPE_TO_DISK_USAGE_SENSOR_SUFFIX.get(cluster.db_type)
    db_type_str = cluster.db_type.name.lower()
    return Alert(
        id="{0}-mdb-disk-usage-{1}".format(db_type_str, cluster.cluster_id),
        project_id=projects.direct.id,
        name="{0} mdb disk usage for cluster {1}(id: {2})".format(
            db_type_str, cluster.name, cluster.cluster_id),
        annotations={
            "host": "direct.prod_{0}_mdb".format(db_type_str),
            "service": "{0}-mdb-disk-space-{1}".format(db_type_str, cluster.name),
            "description": "{{#isAlarm}}" +
                           "{0} cluster {1}(id: {2}) disk usage is {{expression.agv_usage_by_hosts}}".format(
                               db_type_str, cluster.name, cluster.cluster_id) +
                           "warn border {0}, crit border {1}".format(WARN_DISK_BORDER, CRIT_DISK_BORDER) +
                           "{{/isAlarm}}"
        },
        description="Использование диска для кластера {0}(id: {1}) базы {2} в mdb не превышает порогов: "
                    "warn {3}%, crit {4}%".format(cluster.name,
                                                  cluster.cluster_id,
                                                  db_type_str,
                                                  WARN_DISK_BORDER,
                                                  CRIT_DISK_BORDER),
        type=Type(
            expression=Expression(
                program="""
                    let used = {sensor_used};
                    let total = {sensor_total};
                    let used_perc = used / total * 100;
                    let agv_usage_by_hosts = avg(series_max(used_perc));
                    warn_if(agv_usage_by_hosts > {warn_limit});
                    alarm_if(agv_usage_by_hosts > {crit_limit});
            """.format(
                    sensor_used=get_disk_sensor(
                        cluster, "disk-used_bytes_/var/lib/{0}".format(sensor_suffix)),
                    sensor_total=get_disk_sensor(
                        cluster, "disk-total_bytes_/var/lib/{0}".format(sensor_suffix)),
                    warn_limit=WARN_DISK_BORDER, crit_limit=CRIT_DISK_BORDER
                )
            )

        ),
        notification_channels={channels.direct_juggler.id},
        window_secs=10 * 60,
        delay_seconds=0,
    )


def get_disk_sensor(cluster, sensor_name):
    return Sensor(project="internal-mdb", cluster="mdb_{0}".format(cluster.cluster_id), service="mdb",
                  host="*", sensor=sensor_name, dc="by_host",
                  name=sensor_name, node="by_host"
                  )


def create_juggler_aggregates(db_type, clusters, alerts):
    db_type_str = str(db_type.name).lower()
    juggler_host = "direct.prod_{0}_mdb".format(db_type_str)
    export = []
    for cluster in clusters:
        db_aggregate = Check(
            namespace="direct.prod",
            host=juggler_host,
            service=cluster.name,
            refresh_time=90, ttl=900,
            aggregator="logic_or",
            children=[create_child_check_for_alert(alert) for alert in alerts]
        )
        export += alerts
        export.append(db_aggregate)

    service = "{0}_mdb".format(db_type_str)
    aggregate = Check(
        namespace="direct.prod",
        host=juggler_host,
        service=service,
        refresh_time=90, ttl=900,
        aggregator="logic_or",
        children=[
            create_child_check_for_cluster(
                juggler_host,
                cluster) for cluster in clusters]
    )
    export.append(aggregate)
    return export


def create_child_check_for_cluster(host, cluster):
    return Child(
        host=host,
        service=cluster.name,
        group_type="HOST"
    )


def create_child_check_for_alert(alert):
    return Child(
        host=alert.annotations["host"],
        service=alert.annotations["service"],
        group_type="HOST"
    )
