from library.python.monitoring.solo.objects.solomon.v2 import MultiAlert, Alert, Type, Expression

from direct.solo.registered.channel import channels
from direct.solo.registered.project import projects
from direct.solo.registered.sensors.mdb import mysql_threads_running, mysql_threads_connected, mysql_replication_lag

from ..common import MdbCluster
from ..common import DbType


WARN_DISK_BORDER = 90
CRIT_DISK_BORDER = 95
DB_MYSQL_ENVIRON = {
    'production': 'prod'
}

RUNNING_THREADS_WARN_THRESHOLD = 50
RUNNING_THREADS_CRIT_THRESHOLD = 80

CONNECTED_THREADS_WARN_THRESHOLD = 4000
CONNECTED_THREADS_CRIT_THRESHOLD = 6400
CONNECTED_THREADS_PPCDICT_WARN_THRESHOLD = 8000
CONNECTED_THREADS_PPCDICT_CRIT_THRESHOLD = 10000

# еще есть check-db-availability, который делает проверки *fresh_slave* по heartbeat
# порог записан в alldb-config.json
REPLICATION_LAG_THRESHOLD = 300


class MdbClusterMysql(MdbCluster):

    def __init__(
            self,
            cluster_id,
            name,
            instance=None,
            environ=None,
            cgroup=None):
        super().__init__(DbType.MYSQL, cluster_id, name)
        self.cgroup = cgroup
        self.db_type_str = self.db_type.name.lower()
        if instance is None:
            self.instance = name.split('-')[0]
        else:
            self.instance = instance
        if environ is None:
            params = name.split('-')
            if len(params) > 1:
                self.environ = DB_MYSQL_ENVIRON.get(params[1], "UNKNOWN")
            else:
                self.environ = "UNKNOWN"

    # out: ppcdata1,ppcdata2,ppcdata3
    def get_instance_name(self):
        return self.instance

    def get_instance_shortname(self):
        if "ppcdata" in self.instance:
            num = ""
            num = [i for i in self.instance if i.isdigit()]
            return "ppc{0}".format(''.join(num))
        if "ppcmonitor" in self.instance:
            return "monitor"
        return self.instance

    def get_environ(self):
        return self.environ

    def create_freespace(self):
        urlfmt = '''https://solomon.yandex-team.ru/?project=internal-mdb&cluster=mdb_{cluster}&service=mdb&host={{{{labels.host}}}}&node=by_host&sensor=disk-free_bytes_/var/lib/mysql&node=by_host&graph=auto&threshold={{{{expression.borderr}}}}&stack=false'''  # noqa: E501
        return MultiAlert(
            project_id=projects.direct.id,
            id="{2}-{0}-{1}-freespace".format(self.instance,
                                              self.environ, self.db_type_str),
            name="Мониторинг на количество свободного места в MySQL (mdb/{0})".format(
                self.instance),
            annotations={
                "jugglerHost": "{{labels.host}}",
                "jugglerService": "freespace_{0}".format(self.instance),
                "jugglerDescription":
                    "Использовано более {{expression.free_perc_int}}% дисковой квоты. Осталось {{expression.avg_free}}G из {{expression.avg_total}}G. Пороги warn/crit: {{expression.warn_threshold}}%/{{expression.alarm_threshold}}%",  # noqa: E501
                "url": urlfmt.format(cluster=self.cluster_id)
            },
            description='Значение свободного места ниже порогового ({0}): {{{{expression.free_perc_int}}}}'.format(
                self.instance),
            type=Type(
                expression=Expression(
                    program='''
                        let used = group_lines('max', {{project="internal-mdb", cluster="mdb_{cluster}", service="mdb", host="*", node="by_host", sensor="disk-used_bytes_/var/lib/mysql", node="by_host"}});
                        let total = group_lines('max', {{project="internal-mdb", cluster="mdb_{cluster}", service="mdb", host="*", node="by_host", sensor="disk-total_bytes_/var/lib/mysql", node="by_host"}});
                        let free = group_lines('min', {{project="internal-mdb", cluster="mdb_{cluster}", service="mdb", host="*", node="by_host", sensor="disk-free_bytes_/var/lib/mysql", node="by_host"}});
                        let avg_free = round(avg(free) / 1000000000);
                        let avg_total = round(avg(total) / 1000000000);
                        let free_perc = avg(used / total * 100);
                        let free_perc_int = round(free_perc);
                        let alarm_threshold = {crit_limit};
                        let warn_threshold = {warn_limit};
                        warn_if(free_perc > warn_threshold);
                        alarm_if(free_perc > alarm_threshold);
                    '''.format(cluster=self.cluster_id, warn_limit=WARN_DISK_BORDER, crit_limit=CRIT_DISK_BORDER)  # noqa: E501
                )
            ),
            notification_channels={channels.direct_juggler_annotated_full.id},
            window_secs=5 * 60,
            delay_seconds=0,
            group_by_labels={"host"},
        )

    def create_geo_position_master(self):
        urlfmt = '''https://solomon.yandex-team.ru/?project=internal-mdb&cluster={{{{labels.cluster}}}}&service=mdb&l.sensor={{{{labels.sensor}}}}&host={{{{labels.host}}}}&node={{{{labels.node}}}}&graph=auto&stack=false'''  # noqa: E501
        jugglerHost = "direct.prod_ppcdata" if self.instance.count(
            "ppcdata") else "direct.prod_mysql"
        jugglerHost = "direct.prod_xtradb" if self.instance.count(
            "ppcdict") else jugglerHost

        return Alert(
            project_id=projects.direct.id,
            id="{2}-{0}-{1}-geo-position-master".format(self.instance,
                                                        self.environ, self.db_type_str),
            name="Geo position master MySQL (mdb/{0})".format(self.instance),
            annotations={
                "jugglerHost": jugglerHost,
                "jugglerService":
                    "geo_position_master_{0}".format(self.instance),
                "jugglerDescription":
                    "{{#isOk}}мастер в Москве. Переключать не требуется{{/isOk}}{{^isOk}}Мастер в Финке(MAN). Требуется переключить.{{/isOk}}",
                "url": urlfmt.format(cluster=self.cluster_id)
            },
            description='Определяем, что мастер {0} MySQL находится не в MAN(финка)'.format(
    self.instance),
            resolved_empty_policy='RESOLVED_EMPTY_OK',
            no_points_policy='NO_POINTS_DEFAULT',
            type=Type(
                expression=Expression(
                    program='''
                        let threadsConnected = {{project="internal-mdb", cluster="mdb_{cluster}", service="mdb", host="man*", sensor="mysql_role", node="by_host"}};
                        let alarmConnected = map(threadsConnected, line -> avg(line) == 1);
                        let trafficColor = any_of(flatten(alarmConnected)) ? "red" : "green";
                        alarm_if(any_of(flatten(alarmConnected)));
                    '''.format(cluster=self.cluster_id)  # noqa: E501
                )
            ),
            notification_channels={channels.direct_juggler_annotated_full.id},
            window_secs=5 * 60,
            delay_seconds=0,
        )

    def create_running_threads(self):
        sensor = mysql_threads_running.mutate(cluster='mdb_' + self.cluster_id)
        return MultiAlert(
            project_id=projects.direct.id,
            id="{2}-{0}-{1}-threads-running-mdb".format(self.instance,
                                                        self.environ, self.db_type_str),
            name="Мониторинг на количество running threads у MySQL (mdb/{0})".format(
                self.instance),
            annotations={
                "jugglerHost": "{{labels.host}}",
                "jugglerService": "threads_running_{0}".format(self.instance),
                "jugglerDescription":
                    "{{#isOk}}низкое{{/isOk}}{{^isOk}}подозрительно высокое{{/isOk}} значение threads running за 5 минут",
                "url": sensor.mutate(host='{{labels.host}}').build_sensor_link(parameters={'stack': 'false'}),
            },
            description="Мониторинг на количество running threads у MySQL (mdb/{0})".format(self.instance),
            type=Type(
                expression=Expression(
                    program='''
                        let threadsRunning = {selectors};
                        let alarmRunning = map(threadsRunning, line -> avg(line) > {crit});
                        let warnRunning = map(threadsRunning, line -> avg(line) > {warn});

                        let trafficColor = any_of(flatten(alarmRunning)) ? "red" : (any_of(flatten(warnRunning)) ? "yellow" : "green");
                        alarm_if(any_of(flatten(alarmRunning)));
                        warn_if(any_of(flatten(warnRunning)));
                    '''.format(
                        selectors=sensor.selectors,
                        warn=RUNNING_THREADS_WARN_THRESHOLD,
                        crit=RUNNING_THREADS_CRIT_THRESHOLD,
                    )
                )
            ),
            notification_channels={channels.direct_juggler_annotated_full.id},
            window_secs=5 * 60,
            delay_seconds=0,
            group_by_labels={"host"},
        )

    def create_connecting_threads(self):
        sensor = mysql_threads_connected.mutate(cluster='mdb_' + self.cluster_id)
        warn = CONNECTED_THREADS_PPCDICT_WARN_THRESHOLD if self.instance.count('ppcdict') else CONNECTED_THREADS_WARN_THRESHOLD
        crit = CONNECTED_THREADS_PPCDICT_CRIT_THRESHOLD if self.instance.count('ppcdict') else CONNECTED_THREADS_CRIT_THRESHOLD
        return MultiAlert(
            project_id=projects.direct.id,
            id="{2}-{0}-{1}-threads-connected".format(self.instance,
                                                      self.environ, self.db_type_str),
            name="Мониторинг на количество connected threads у MySQL (mdb/{0})".format(self.instance),
            annotations={
                "jugglerHost": "{{labels.host}}",
                "jugglerService":
                    "threads_connected_{0}".format(self.instance),
                "jugglerDescription":
                    "{{#isOk}}низкое{{/isOk}}{{^isOk}}подозрительно высокое{{/isOk}} значение threads connected за 5 минут",
                "url": sensor.mutate(host='{{labels.host}}').build_sensor_link(parameters={'stack': 'false'}),
            },
            description="Мониторинг на количество connected threads у MySQL (mdb/{0})".format(
                self.instance),
            type=Type(
                expression=Expression(
                    program='''
                        let threadsConnected = {selectors};
                        let alarmConnected = map(threadsConnected, line -> max(line) > {crit});
                        let warnConnected = map(threadsConnected, line -> max(line) > {warn});

                        let trafficColor = any_of(flatten(alarmConnected)) ? "red" : (any_of(flatten(warnConnected)) ? "yellow" : "green");
                        alarm_if(any_of(flatten(alarmConnected)));
                        warn_if(any_of(flatten(warnConnected)));
                    '''.format(selectors=sensor.selectors, warn=warn, crit=crit)
                )
            ),
            notification_channels={channels.direct_juggler_annotated_full.id},
            window_secs=5 * 60,
            delay_seconds=0,
            group_by_labels={"host"},
        )

    def create_replication_lag(self):
        sensor=mysql_replication_lag.mutate(cluster='mdb_' + self.cluster_id)
        return MultiAlert(
            project_id=projects.direct.id,
            id="{2}-{0}-{1}-replication-lag".format(self.instance,
                                                    self.environ, self.db_type_str),
            name="Мониторинг на отставание реплик MySQL (mdb/{0})".format(self.instance),
            annotations={
                "jugglerHost": "{{labels.host}}",
                "jugglerService":
                    "mysql_slave_behind_{0}".format(self.instance),
                "jugglerDescription":
                    "{{#isOk}}нет отставания{{/isOk}}{{^isOk}}высокое отставание реплики{{/isOk}} значение replication_lag за 5 минут",
                "url": sensor.mutate(host='{{labels.host}}').build_sensor_link(parameters={'stack': 'false'}),
            },
            description="Мониторинг на отставание реплик MySQL (mdb/{0})".format(self.instance),
            type=Type(
                expression=Expression(
                    program='''
                        let threadsLag = {selectors};
                        let alarmLag = map(threadsLag, line -> avg(line) > {crit});

                        let trafficColor = any_of(flatten(alarmLag)) ? "red" : "green";
                        alarm_if(any_of(flatten(alarmLag)));
                    '''.format(
                        selectors=sensor.selectors,
                        crit=REPLICATION_LAG_THRESHOLD,
                    )
                )
            ),
            notification_channels={channels.direct_juggler_annotated_full.id},
            window_secs=5 * 60,
            delay_seconds=0,
            group_by_labels={"host"},
        )

    def create_ready_for_operation(self):
        urlfmt = '''https://solomon.yandex-team.ru/?project=internal-mdb&cluster={{{{labels.cluster}}}}&service=mdb&l.sensor={{{{labels.sensor}}}}&host={{{{labels.host}}}}&node={{{{labels.node}}}}&graph=auto&stack=false'''  # noqa: E501
        return MultiAlert(
            project_id=projects.direct.id,
            id="{2}-{0}-{1}-ready-for-operation".format(self.instance,
                                                        self.environ, self.db_type_str),
            name="Ready for operation MySQL инстанса (mdb/{0})".format(
                self.instance),
            annotations={
                "jugglerHost": "{{labels.host}}",
                "jugglerService": "ready_for_operation",
                "jugglerDescription":
                    "{{#isOk}}хост живой{{/isOk}}{{^isOk}}хост умер{{/isOk}}",
                "url": urlfmt.format(cluster=self.cluster_id)
            },
            description="Ready for operation MySQL инстанса (mdb/{0})".format(
    self.instance),
            no_points_policy="NO_POINTS_NO_DATA",
            type=Type(
                expression=Expression(
                    program='''
                        let threadsConnected = {{project="internal-mdb", cluster="mdb_{cluster}", service="mdb", host="*", sensor="mysql_role", node="by_host"}};
                        let alarmConnected = map(threadsConnected, line -> avg(line) < 1);
                        let trafficColor = any_of(flatten(alarmConnected)) ? "red" : "green";
                        alarm_if(any_of(flatten(alarmConnected)));
                    '''.format(cluster=self.cluster_id)  # noqa: E501
                )
            ),
            notification_channels={channels.direct_juggler_annotated_full.id},
            window_secs=5 * 60,
            delay_seconds=0,
            group_by_labels={"host", "sensor"},
        )


'''
        host: direct.prod_ppcdata
        service: freespace_ppcdata10
        refresh_time: 90
        ttl: 900
        aggregator: logic_or
        tags:
                - directadmin_TV
                - direct_group_sre
                - a_mark_direct.prod_ppcdata
        children:
                -
                host: direct.prod_ppcdata.db_mdbdqj1mh9vb2gs2rk94
                service: freespace_ppcdata10
                type: HOST
        notifications:
                -
                description: сервис для хранения истории по проверкам
        template_kwargs:
            push_url: 'http://juggler-history.da.yandex.ru/save_notifications'
        template_name: push

        host: direct.prod_ppcdata.db_mdbdqj1mh9vb2gs2rk94
        service: freespace_ppcdata10
        refresh_time: 90
        ttl: 900
        aggregator: logic_or
        aggregator_kwargs:
        unreach_mode: skip
        unreach_service:
                -
                check: 'direct.prod_ppcdata.db_mdbdqj1mh9vb2gs2rk94:ready_for_operation'
                hold: 900
        tags:
                - directadmin.aggregation_child
                - a_mark_direct.prod_ppcdata
        children:
                -
                host: db_mdbdqj1mh9vb2gs2rk94
                service: freespace_ppcdata10
                type: CGROUP

    def create_juggler_aggregates(self):
        export = []

        meta = ENVTYPES.get(self.environ)
        for service_name in ['freespace_{}']:
            instance = self.get_instance_name()

        service_name = "direct-mysql-ptkill-{}".format(self.environ)
        tags_name = [
            'direct-subsystems-health',
            'a_mark_direct.prod_subsystems'] if self.environ == 'production' else [
        ]
        aggregate = Check(
            namespace="direct.{}".format(meta.get('envshort')),
            host="direct.{0}_direct_ptkill_{1}".format(
                meta.get('envshort'), self.environ),
            service=service_name,
            refresh_time=90, ttl=900,
            aggregator="logic_and",
            tags=tags_name,
            children=self.create_child_check_for_cluster(
                meta.get('cgroup'), self.clusters, self.environ)
        )
        export.append(aggregate)
        return expor
'''
