from juggler_sdk import Check, Child
from . import resources
from . import health


# Значения должно быть степенью двойки, время в миллисекундах
class CritDuration:
    def __init__(self, read_only, read_write, write_only):
        self.read_only = read_only
        self.read_write = read_write
        self.write_only = write_only


class YdbDatabase:
    def __init__(self, cluster, db_name, crit_durations):
        self.cluster = cluster
        self.db_name = db_name
        self.crit_durations = crit_durations


YDB_DATABASES = [
    YdbDatabase('ydb_ru', '/ru/direct/production/hourglass',
                CritDuration(read_only=512, read_write=1024, write_only=512)),
    YdbDatabase('ydb_ru', '/ru/direct/production/frontdb',
                CritDuration(read_only=512, read_write=512, write_only=512)),
    YdbDatabase('ydb_ru', '/ru/direct/production/maintenance-helpers',
                CritDuration(read_only=65536, read_write=65536, write_only=65536)),
]


def get_all_alerts():
    export = []
    for ydb_database in YDB_DATABASES:
        alerts = [
            resources.get_cpu_alter(ydb_database),
            resources.get_mem_alter(ydb_database),
            health.get_unavailable_alert(ydb_database),
            health.get_hard_errors_alert(ydb_database)]

        alerts.extend(health.get_transaction_duration_alerts(ydb_database))

        db_aggregate = Check(
            namespace="direct.prod",
            host="direct.prod_ydb",
            service=get_ydb_database_service_name(ydb_database),
            refresh_time=90, ttl=900,
            aggregator="logic_or",
            meta={
                "urls":
                    [{
                        "title": "Дашборд для базы {0} в Solomon".format(ydb_database.db_name),
                        "url": "https://solomon.yandex-team.ru/?project=kikimr&service=kqp&host=cluster&slot=static&"
                               "dashboard=multitenant-client-database-kqp&database={0}&cluster={1}".format(
                            ydb_database.db_name, ydb_database.cluster)
                    }]
            },
            children=[create_child_check_for_alert(alert) for alert in alerts]
        )
        export += alerts
        export.append(db_aggregate)

    ydb_aggregate = Check(
        namespace="direct.prod",
        host="direct.prod_ydb",
        service="ydb",
        refresh_time=90, ttl=900,
        aggregator="logic_or",
        children=[create_child_check_for_db(ydb_database) for ydb_database in YDB_DATABASES]
    )
    export.append(ydb_aggregate)
    return export


def get_ydb_database_service_name(ydb_database):
    return "{0}".format(ydb_database.db_name)


def create_child_check_for_db(ydb_database):
    return Child(
        host="direct.prod_ydb",
        service=get_ydb_database_service_name(ydb_database),
        group_type="HOST"
    )


def create_child_check_for_alert(alert):
    return Child(
        host=alert.annotations["host"],
        service=alert.annotations["service"],
        group_type="HOST"
    )
