from enum import Enum

from library.python.monitoring.solo.objects.solomon.v2 import Alert, Type, Expression
from library.python.monitoring.solo.objects.solomon.sensor import Sensor
from direct.solo.registered.channel import channels
from direct.solo.registered.project import projects

WARN_LIMIT = 5
CRIT_LIMIT = 10
ERR_DOC_LINK = 'https://ydb.yandex-team.ru/docs/troubleshooting/diag_dashboard#request-errors'
QUERIES_DOC_LINK = 'https://ydb.yandex-team.ru/docs/troubleshooting/system_views#topy-zaprosov'
HARD_ERRORS = 'YdbResponses/Unsupported|' \
              'YdbResponses/Unspecified|' \
              'YdbResponses/Undetermined|' \
              'YdbResponses/Timeout|' \
              'YdbResponses/SchemeError|' \
              'YdbResponses/PreconditionFailed|' \
              'YdbResponses/Other|' \
              'YdbResponses/NotFound|' \
              'YdbResponses/InternalError|' \
              'YdbResponses/GenericError|' \
              'YdbResponses/BadRequest|' \
              'YdbResponses/AlreadyExists'

TRANSACTION_DURATIONS_METRIC = [1 << i for i in range(18)]


class TransactionType(Enum):
    READ_ONLY = 1
    READ_WRITE = 2
    WRITE_ONLY = 3


TRANSACTION_TYPE_TO_DURATION_SENSOR = {
    TransactionType.READ_ONLY: 'Transactions/TotalDuration/ReadOnly',
    TransactionType.READ_WRITE: 'Transactions/TotalDuration/ReadWrite',
    TransactionType.WRITE_ONLY: 'Transactions/TotalDuration/WriteOnly',
}


def get_hard_errors_alert(ydb_database):
    return Alert(
        id="ydb-hard-errors-{0}-{1}".format(ydb_database.cluster, ydb_database.db_name.replace("/", "-")),
        project_id=projects.direct.id,
        name="ydb-hard-errors {0}:{1}".format(ydb_database.cluster, ydb_database.db_name),
        annotations={
            "host": "direct-ydb",
            "service": "ydb-hard-errors-{0}-{1}".format(ydb_database.cluster, ydb_database.db_name),
            "description": "Hard errors percent for database-{0}:{1}".format(ydb_database.cluster,
                                                                             ydb_database.db_name) +
                           " is {{expression.agv_ratio}}" +
                           " warn limit={0}, crit limit={1}".format(WARN_LIMIT, CRIT_LIMIT) +
                           " Doc: {0}".format(ERR_DOC_LINK)

        },
        description="Тяжелые ошибки базой {0}:{1}, такие ошибки не ретраятся, "
                    "свидетельствую о проблемах со схемой или запросами"
                    "документация {2}".format(ydb_database.cluster, ydb_database.db_name, ERR_DOC_LINK),
        type=Type(
            expression=Expression(
                program="""
                let hard_errors = {0};
                let ok_responses = {1};

                let ratio = hard_errors/ok_responses * 100;
                let agv_ratio= avg(series_max(ratio));
                warn_if(agv_ratio > {2});
                alarm_if(agv_ratio > {3});
            """.format(
                    Sensor(
                        project="kikimr", cluster=ydb_database.cluster, service="kqp",
                        host="cluster", sensor=HARD_ERRORS, database=ydb_database.db_name, slot="cluster"
                    ),
                    Sensor(
                        project="kikimr", cluster=ydb_database.cluster, service="kqp",
                        host="cluster", sensor="YdbResponses/Success", database=ydb_database.db_name, slot="cluster"
                    ),
                    WARN_LIMIT, CRIT_LIMIT
                )
            )

        ),
        notification_channels={channels.direct_juggler.id},
        window_secs=10 * 60,
        delay_seconds=0,
    )


def get_unavailable_alert(ydb_database):
    return Alert(
        id="ydb-unavailable-{0}-{1}".format(ydb_database.cluster, ydb_database.db_name.replace("/", "-")),
        project_id=projects.direct.id,
        name="ydb-unavailable-errors {0}:{1}".format(ydb_database.cluster, ydb_database.db_name),
        annotations={
            "host": "direct-ydb",
            "service": "ydb-unavailable-errors-{0}-{1}".format(ydb_database.cluster, ydb_database.db_name),
            "description": "Unavailable errors for database-{0}:{1}".format(ydb_database.cluster,
                                                                            ydb_database.db_name) +
                           " is {{expression.agv_ratio}}" +
                           " warn limit={0}, crit limit={1}".format(WARN_LIMIT, CRIT_LIMIT) +
                           " Doc: {0}".format(ERR_DOC_LINK)

        },
        description="Ошибки unavailable для базы {0}:{1}, "
                    "документация {2}".format(ydb_database.cluster, ydb_database.db_name, ERR_DOC_LINK),
        type=Type(
            expression=Expression(
                program="""
                let unavailable_errors = {0};
                let ok_responses = {1};

                let ratio = unavailable_errors/ok_responses * 100;
                let agv_ratio= avg(ratio);
                warn_if(agv_ratio > {2});
                alarm_if(agv_ratio > {3});
            """.format(
                    Sensor(
                        project="kikimr", cluster=ydb_database.cluster, service="kqp",
                        host="cluster", sensor="YdbResponses/Unavailable", database=ydb_database.db_name, slot="cluster"
                    ),
                    Sensor(
                        project="kikimr", cluster=ydb_database.cluster, service="kqp",
                        host="cluster", sensor="YdbResponses/Success", database=ydb_database.db_name, slot="cluster"
                    ),
                    WARN_LIMIT, CRIT_LIMIT
                )
            )

        ),
        notification_channels={channels.direct_juggler.id},
        window_secs=10 * 60,
        delay_seconds=0,
    )


def get_transaction_duration_alerts(ydb_database):
    return [get_transaction_duration_alert_for_type(ydb_database, TransactionType.READ_ONLY,
                                                    ydb_database.crit_durations.read_only),
            get_transaction_duration_alert_for_type(ydb_database, TransactionType.READ_WRITE,
                                                    ydb_database.crit_durations.read_write),
            get_transaction_duration_alert_for_type(ydb_database, TransactionType.WRITE_ONLY,
                                                    ydb_database.crit_durations.write_only)]


def get_transaction_duration_alert_for_type(ydb_database, transaction_type, crit_value):
    transaction_type_str = transaction_type.name.lower()
    all_crit_values = get_transaction_duration_label(filter(lambda d: d >= crit_value, TRANSACTION_DURATIONS_METRIC))
    all_values = get_transaction_duration_label(TRANSACTION_DURATIONS_METRIC)
    sensor = TRANSACTION_TYPE_TO_DURATION_SENSOR[transaction_type]
    return Alert(
        id="ydb-{0}-ms-{1}".format(transaction_type_str, ydb_database.db_name.replace("/", "-")),
        project_id=projects.direct.id,
        name="ydb {0} transactions duration for database {1}".format(transaction_type_str, ydb_database.db_name),
        annotations={
            "host": "direct-ydb",
            "service": "ydb-{0}-transaction-ms-{1}-{2}".format(transaction_type_str, ydb_database.cluster,
                                                               ydb_database.db_name.replace("/", "-")),
            "description": "{0}:{1} Duration of {2} transactions less then {3} ms. Doc how to find bad queries "
                           "{4}".format(ydb_database.cluster, ydb_database.db_name, transaction_type_str, crit_value,
                                        QUERIES_DOC_LINK)

        },
        description="Длительность {0} транзакций для базы {1}:{2} не превышает {3} ms, "
                    "документация, как находить плохие запросы "
                    "{4}".format(transaction_type_str, ydb_database.cluster, ydb_database.db_name, crit_value,
                                 QUERIES_DOC_LINK),
        type=Type(
            expression=Expression(
                program="""
                let crit_duration_values = {0};
                let all_duration_values = {1};

                let long_queries_cnt = series_sum(crit_duration_values);
                let all_queries_cnt = series_sum(all_duration_values);
                let ratio = long_queries_cnt/all_queries_cnt * 100;
                let avg_ratio = avg(ratio);
                alarm_if(avg_ratio >= 1);
            """.format(
                    Sensor(
                        project="kikimr", cluster=ydb_database.cluster, service="kqp",
                        host="cluster", sensor=sensor,
                        database=ydb_database.db_name, slot="cluster",
                        range=all_crit_values
                    ),
                    Sensor(
                        project="kikimr", cluster=ydb_database.cluster, service="kqp",
                        host="cluster", sensor=sensor,
                        database=ydb_database.db_name, slot="cluster",
                        range=all_values
                    )
                )
            )

        ),
        notification_channels={channels.direct_juggler.id},
        window_secs=5 * 60,
        delay_seconds=0,
    )


def get_transaction_duration_label(values):
    values_list = list(map(lambda d: "{0} ms".format(d), values))
    values_list.append("INF ms")
    return "|".join(values_list)
