from juggler_sdk import Check, Child
from library.python.monitoring.solo.objects.solomon.v2 import Alert, Type, Expression

from direct.solo.registered.channel import channels
from direct.solo.registered.juggler import reference
from direct.solo.registered.juggler.reference import get_warden_tags
from direct.solo.registered.project import projects
from direct.solo.registered.sensors.external_metrics import metrika_server_errors


def build_metrika_errors_expression(border):
    return Expression(
        program="""
        let err_rate = {};
        let errors = integral(err_rate);
        let last = map(errors, line -> get_label(line, 'method') + ': ' + to_string(last(line)));
        alarm_if( any_of( map(errors, line -> last(line) > {}) ) );
        """.format(metrika_server_errors.selectors, border)
    )

ERRORS_NOTIFY_BORDER=100
ERRORS_NOTIFY_WINDOW=5
ERRORS_DISASTER_BORDER=1000
ERRORS_DISASTER_WINDOW=10
HOST_SUFFIX="metrika"
SERVICE_SERVER_ERRORS='requests_server_errors_count'

metrika_requests_server_errors = Alert(
    id='metrica_requests_server_errors',
    project_id=projects.direct.id,
    name='Фон ошибок при запросах к Метрике',
    description='Ошибок при запросах в Метрику меньше {} за последние {} минут'.format(
        ERRORS_NOTIFY_BORDER,
        ERRORS_NOTIFY_WINDOW,
    ),
    annotations={
        'alarmMessage': 'Появился фон ошибок при запросах в Метрику.'
            + '\nМогут быть недоступны цели в статистике и при редактировании',
        'okMessage': 'Ошибки при запросах к Метрике — прекратились.',
    },
    window_secs=ERRORS_NOTIFY_WINDOW * 60,
    type=Type(expression=build_metrika_errors_expression(ERRORS_NOTIFY_BORDER)),
    notification_channels={
        channels.direct_duty_chat.id,
    },
    resolved_empty_policy="RESOLVED_EMPTY_NO_DATA",
)

metrika_requests_server_errors_disaster = Alert(
    id='metrica_requests_server_errors_disaster',
    project_id=projects.direct.id,
    name='Ошибки при запросах к Метрике',
    description='Алерт для заведения SPI при превышении числа ошибок запросов к Метрику выше {} за последние {} минут'.format(
        ERRORS_DISASTER_BORDER,
        ERRORS_DISASTER_WINDOW,
    ),
    annotations={
        "jugglerHostSuffix": HOST_SUFFIX,
        "jugglerService": SERVICE_SERVER_ERRORS,
        "jugglerDescription": "Ошибок при запросах в Метрику за последние " + str(ERRORS_DISASTER_WINDOW) + " минут: {{expression.last}}",
    },
    window_secs=ERRORS_DISASTER_WINDOW * 60,
    type=Type(expression=build_metrika_errors_expression(ERRORS_DISASTER_BORDER)),
    notification_channels={
        channels.direct_juggler_annotated.id,
    },
    resolved_empty_policy="RESOLVED_EMPTY_NO_DATA",
    delay_seconds=0,
)

metrika_requests_server_errors_check = Check(
    namespace=reference.NAMESPACE_PROD,
    host='direct.disaster_alerts',
    service="metrika_requests_server_errors",
    aggregator="logic_or",
    ttl=600,
    refresh_time=60,
    children=[
        Child(host=channels.get_solomon_host(HOST_SUFFIX), service=SERVICE_SERVER_ERRORS),
    ],
    tags=get_warden_tags(slug='direct__metrika-integration'),
    meta={
        "urls": [
            {
                "type": "yasm_alert",
                "title": "rate ошибок",
                "url": metrika_server_errors.build_sensor_link({'cs': 'default', 'stack': 'false', 'b': '6h'}),
            }, {
                "type": "wiki",
                "title": "документация",
                "url": "https://docs.yandex-team.ru/direct-dev/reference/alerts/metrika-requests-server-errors",
            }
        ]
    },
    notifications=[
        reference.JUGGLER_HISTORY,
    ],
)


exports=[
    metrika_requests_server_errors,
    metrika_requests_server_errors_disaster,
    metrika_requests_server_errors_check,
]
