from library.python.monitoring.solo.objects.solomon.v2 import MultiAlert, Type, Expression
from library.python.monitoring.solo.objects.solomon.sensor import Sensor
from direct.solo.registered.channel import channels
from direct.solo.registered.project import projects

WINDOW_MINUTES = 10  # Длительность измерения в минутах
MIN_ERR_COUNT_LIMIT = 50  # Пороговое количество ошибок для вычисления алёрта
ERR_LIMIT_PERC = 1  # Доля ошибок, на которую реагируем
# Доля точек, для которых количество ошибок превышает порог ERR_LIMIT_PERC
ERR_RATIO = {
    "WARN": 0.2,
    "CRIT": 0.5,
}


def get_sprav_sensor(sensor):
    return Sensor(
        project="direct", cluster="app_java-web|app_java-intapi|app_java-api5", service="java-monitoring",
        host="CLUSTER", parallel_fetcher="organizations_client:*", env="production", sensor=sensor
    )


sprav_requests_server_errors = MultiAlert(
    id="sprav_requests_server_errors",
    project_id=projects.direct.id,
    name="Фон ошибок при запросах к Справочнику",
    description="""
Алерт проверяет уровень ошибок на запросы Справочника из Директа в разных приложениях (web, intapi, api5).
Замеры проверятся по solomon-метрикам Директа.

Алёрт не зажигается, если за последние {window} минут получили меньше {min_err_count_limit} ошибок.
Если ошибок больше, считаем, какую долю времени они превышают порог в {err_limit_perc}%. Если больше {crit_err_ratio}, зажигаем CRIT.

Если ошибки появились только в одном приложении, про проблема могла прийти с релизом.
Если ошибки подскочили во всех сервисах, то могут быть проблемы Справочника. Логи с запросами, ответами и ошибками можно смотреть тут: https://direct.yandex.ru/logviewer/short/_DSDD39b3YXysj.
Призывать коллег из Справочника можно в чате "Tycoon-support" https://t.me/joinchat/BuMzDk_uMG_FuZpoQFEo4w.
        """.format(
        window=WINDOW_MINUTES,
        min_err_count_limit=MIN_ERR_COUNT_LIMIT,
        err_limit_perc=ERR_LIMIT_PERC,
        crit_err_ratio=ERR_RATIO["CRIT"],
    ),
    group_by_labels={"cluster"},
    window_secs=WINDOW_MINUTES * 60 + 300,
    delay_seconds=0,
    notification_channels={channels.direct_juggler_annotated.id},
    annotations={
        "points_to_analyze": "{{expression.points_to_analyze}}",
        "warn_count_limit": "{{expression.warn_count_limit}}",
        "crit_count_limit": "{{expression.crit_count_limit}}",
        "err_limit_perc": "{{expression.err_limit_perc}}",
        "err_exceeded_count": "{{expression.err_exceeded_count}}",
        "total_err_count": "{{expression.total_err_count}}",
        "skip_low_impact": "{{expression.skip_low_impact}}",
        "jugglerHostSuffix": "{{expression.jugglerHostSuffix}}",
        "jugglerService": "sprav",
        "url": "https://solomon.yandex-team.ru/admin/projects/direct/alerts/{{alert.id}}",
        "jugglerDescription": "{{#isOk}}OK{{/isOk}}{{^isOk}}Доля ошибок выше {{expression.err_limit_perc}}% для {{expression.err_exceeded_count}} из {{expression.points_to_analyze}} точек{{/isOk}}",
        "label": "{{expression.label}}",
    },
    type=Type(
        expression=Expression(
            program="""
                let requests = sum({requests}) by cluster;
                let errors = sum({failures}) by cluster;

                let label = get_label(requests, "cluster");
                let jugglerHostSuffix = label == "app_java-intapi" ? "java-intapi" : (label == "app_java-web" ? "java-web" : (label == "app_java-api5" ? "java-api" : "other"));

                let err_limit_perc = {err_limit_perc};
                let min_err_count_limit = {min_err_count_limit};
                let window = {window_minutes}m;

                // Доля точек выше порога для зажигания алёрта
                let warn_err_ratio = {warn_err_ratio};
                let crit_err_ratio = {crit_err_ratio};

                // одну точку выкидываем для красоты. 10m -- это 41 точка, например
                let points_to_analyze = count(tail(requests, window)) - 1;
                let warn_count_limit = points_to_analyze * warn_err_ratio;
                let crit_count_limit = points_to_analyze * crit_err_ratio;

                let total_err_count = sum(tail(errors, points_to_analyze)) * 15; // 15s -- шаг сетки. Домножаем, чтобы получить абсолюты
                let skip_low_impact = total_err_count < min_err_count_limit;

                // Количество точек, для которых доля ошибок выше err_limit_perc
                let err_exceeded_count = count(drop_below(100.0 * tail(errors, points_to_analyze) / tail(requests, points_to_analyze), err_limit_perc));

                let is_crit = !skip_low_impact && (err_exceeded_count > crit_count_limit);
                alarm_if(is_crit);
                let is_warn = !skip_low_impact && (err_exceeded_count > warn_count_limit);
                warn_if(is_warn);
                """.format(
                requests=get_sprav_sensor("requests"),
                failures=get_sprav_sensor("failures"),
                err_limit_perc=ERR_LIMIT_PERC,
                min_err_count_limit=MIN_ERR_COUNT_LIMIT,
                warn_err_ratio=ERR_RATIO["WARN"],
                crit_err_ratio=ERR_RATIO["CRIT"],
                window_minutes=WINDOW_MINUTES,
            )
        )
    ),
)

exports=[sprav_requests_server_errors]
