from juggler_sdk import Check, Child, NotificationOptions
from library.python.monitoring.solo.objects.solomon.sensor import Sensor
from library.python.monitoring.solo.objects.solomon.v2 import MultiAlert, Type, Expression

from direct.solo.registered import config
from direct.solo.registered.channel import channels
from direct.solo.registered.juggler import reference
from direct.solo.registered.project import projects

yt_pools = "|".join(config.YT_POOLS)
NOTIFY_WINDOW = 60
DELAY = 30

yt_operations_limit = MultiAlert(
    id='yt_operations_limit',
    project_id=projects.direct.id,
    name='Limit YT operations for direct_all (Solo)',
    description='Большое количество операций. Более подробно можно посмотреть во внутреннем инструменте https://direct.yandex.ru/internal_tools/#yql_resources_stat',
    window_secs=NOTIFY_WINDOW * 60,
    delay_seconds=0,
    group_by_labels={"pool", "cluster"},
    annotations={
        "jugglerHostSuffix": "{{expression.juggler_host_suffix}}",
        "jugglerService": "yt_operations_limit",
        "jugglerDescription": "Для пула {{labels.pool}} уперлись в лимит запросов на кластере {{labels.cluster}}",
        "url": "https://solomon.yandex-team.ru/admin/projects/direct/alerts/{{alert.id}}"
    },
    type=Type(
        expression=Expression(
            program="""
            let op_count = {op_count_sensor};
            let op_max = {op_max_sensor};

            let pool = get_label(op_count, "pool");
            let need_notify = pool == "robot-direct-yt"; // Здесь можно указать другие пулы, по которым тоже нужна нотификация
            let juggler_host_suffix = need_notify ? "yt_notify" : "yt_not_notify";

            let op_frac = avg(op_count)/avg(op_max);

            let is_red = op_frac >= 0.95;
            let trafficColor = is_red ? 'red' : 'green';

            alarm_if(trafficColor == 'red');
            ok_if(trafficColor == 'green');
            """.format(
                op_count_sensor=Sensor(
                    project="yt", cluster="*", service="scheduler_pools",
                    sensor="yt.scheduler.pools.total_operation_count",
                    pool=yt_pools, tree="physical|cloud"
                ),
                op_max_sensor=Sensor(
                    project="yt", cluster="*", service="scheduler_pools",
                    sensor="yt.scheduler.pools.max_operation_count",
                    pool=yt_pools, tree="physical|cloud"
                ),
            )
        ),
    ),
    notification_channels={channels.direct_juggler_annotated.id},
    resolved_empty_policy="RESOLVED_EMPTY_NO_DATA",
)

yt_operations_limit_check = Check(
    namespace=reference.NAMESPACE_DEV,
    host="direct.prod_common",
    service="yt_operations_limit",
    aggregator="logic_or",
    ttl=900,
    refresh_time=300,
    children=[
        Child(host=channels.get_solomon_host("yt_notify"), service="yt_operations_limit"),
    ],
    notifications=[
        NotificationOptions(
            template_name='on_status_change',
            template_kwargs={
                'status': ["OK", "WARN", "CRIT"],
                'login': ['kalchevskaya'],
                'method': ['telegram'],
                'delay': 900
            }
        )
    ],
)

exports = [
    yt_operations_limit,
    yt_operations_limit_check,
]
