from juggler_sdk import Check, Child
from library.python.monitoring.solo.objects.solomon.v2 import Alert, Type, PredicateRule, Threshold

from direct.solo.registered.channel import channels
from direct.solo.registered.juggler import reference
from direct.solo.registered.project import projects
from direct.solo.registered.sensors.ess import aggregated_statuses_delay, aggregated_statuses_read_lag_ms
from direct.solo.registered.juggler.reference import CALLING_CHECKS_CHAT
from direct.solo.registered.juggler.reference import LIGHT_DUTY_ESCALATION

WARN_DELAY = 30 * 60
CRIT_DELAY = 40 * 60

HOST_SUFFIX = "aggr_statuses"
SERVICE_PROCESSOR_DELAY = "processor_delay"
SERVICE_TOPIC_READ_DELAY = "topic_read_delay"

processor_delay_alert = Alert(
    id="aggregated_statuses_delay_sec",
    project_id=projects.direct.id,
    name="aggregated_statuses_delay_sec # solo",
    annotations={
        "jugglerHostSuffix": HOST_SUFFIX,
        "jugglerService": SERVICE_PROCESSOR_DELAY,
        "jugglerDescription": "Максимальное отставание в обработке агрегированных статусов {{#isOk}}в норме{{/isOk}}" +
                              "{{^isOk}}превышает {{#isAlarm}}" + str(CRIT_DELAY) + "{{/isAlarm}}" +
                              "{{#isWarn}}" + str(WARN_DELAY) + "{{/isWarn}} секунд{{/isOk}}",
        "trafficLight": "{{expression.trafficColor}}",
    },
    type=Type(
        threshold=Threshold(
            selectors=aggregated_statuses_delay.selectors,
            time_aggregation="LAST_NON_NAN",
            predicate="GT",
            threshold=CRIT_DELAY,
            predicate_rules=[
                PredicateRule(
                    threshold_type="LAST_NON_NAN",
                    comparison="GT",
                    threshold=CRIT_DELAY,
                    target_status="ALARM"
                ),
                PredicateRule(
                    threshold_type="LAST_NON_NAN",
                    comparison="GT",
                    threshold=WARN_DELAY,
                    target_status="WARN"
                ),
            ],
        ),
    ),
    notification_channels={
        channels.direct_juggler_annotated.id
    },
    window_secs=10 * 60,
    delay_seconds=0,
)

read_topic_delay = Alert(
    id="aggregated_statuses_read_topic_delay_ms",
    project_id=projects.direct.id,
    name="aggregated_statuses_read_topic_delay_ms # solo",
    annotations={
        "jugglerHostSuffix": HOST_SUFFIX,
        "jugglerService": SERVICE_TOPIC_READ_DELAY,
        "jugglerDescription": "Отставание вычитки событий агрегированных статусов из logbroker {{#isOk}}в норме{{/isOk}}" +
                              "{{^isOk}}превышает {{#isAlarm}}" + str(CRIT_DELAY) + "{{/isAlarm}}" +
                              "{{#isWarn}}" + str(WARN_DELAY) + "{{/isWarn}} секунд{{/isOk}}",
        "trafficLight": "{{expression.trafficColor}}",
    },
    type=Type(
        threshold=Threshold(
            selectors=aggregated_statuses_read_lag_ms.selectors,
            time_aggregation="LAST_NON_NAN",
            predicate="GT",
            threshold=CRIT_DELAY * 1000,  # to millis
            predicate_rules=[
                PredicateRule(
                    threshold_type="LAST_NON_NAN",
                    comparison="GT",
                    threshold=CRIT_DELAY * 1000,  # to millis
                    target_status="ALARM"
                ),
                PredicateRule(
                    threshold_type="LAST_NON_NAN",
                    comparison="GT",
                    threshold=WARN_DELAY * 1000,  # to millis
                    target_status="WARN"
                ),
            ],
        ),
    ),
    notification_channels={
        channels.direct_juggler_annotated.id
    },
    window_secs=10 * 60,
    delay_seconds=0,
)

aggr_statuses_check = Check(
    namespace=reference.NAMESPACE_PROD,
    host='direct.disaster_alerts',
    service="aggr_statuses_delay",
    aggregator="logic_or",
    ttl=600,
    refresh_time=60,
    children=[
        Child(host=channels.get_solomon_host(HOST_SUFFIX), service=SERVICE_PROCESSOR_DELAY),
        Child(host=channels.get_solomon_host(HOST_SUFFIX), service=SERVICE_TOPIC_READ_DELAY),
    ],
    tags=[
        'warden_auto_source',
        'warden_functionality_direct__aggr_statuses_freshness',
        'warden_alert_category_boolean',
    ],
    meta={
        "urls": [
            {
                "type": "yasm_alert",
                "title": "отставание по шардам",
                "url": aggregated_statuses_delay.build_sensor_link({'b': '3h', 'stack': 'false'})
            },
            {
                "type": "wiki",
                "title": "документация",
                "url": "https://docs.yandex-team.ru/direct-dev/aggr-statuses/jeri"
            },
        ]
    },
    notifications=[
        LIGHT_DUTY_ESCALATION,
        CALLING_CHECKS_CHAT,
    ],
)

exports = [
    processor_delay_alert,
    read_topic_delay,
    aggr_statuses_check,
]
