from juggler_sdk import Check, Child, FlapOptions
from library.python.monitoring.solo.objects.solomon.v2 import Alert, Type, Expression

from direct.solo.registered.channel.channels import get_solomon_host, direct_juggler_annotated
from direct.solo.registered.juggler.reference import NAMESPACE_PROD, JUGGLER_HISTORY, get_warden_tags
from direct.solo.registered.sensors.jobs import threads_running, threads_count

CRIT_POOL_USAGE_PCT = 90
DCS_COUNT = 4   # число ДЦ
HOST_SUFFIX = 'direct-jobs'


def get_service(env):
    return 'thread_pool_usage.' + env


def create_pool_alert(env):
    running = threads_running.mutate(env=env)
    size = threads_count.mutate(env=env)
    link_sensor = running.mutate(sensor='{a}|{b}'.format(a=threads_running.labels['sensor'], b=threads_count.labels['sensor']))

    return Alert(
        id="hourglass_pool_usage_" + env,
        project_id=running.labels['project'],
        name="Jobs thread pool usage in {env}".format(env=env),
        annotations={
            "jugglerHostSuffix": HOST_SUFFIX,
            "jugglerService": get_service(env),
            "jugglerDescription": "Утилизация тред-пула {{#isOk}}в норме{{/isOk}}"
                + "{{^isOk}}{{#isAlarm}}превышает " + str(CRIT_POOL_USAGE_PCT) + "%{{/isAlarm}}"
                + "{{#isWarn}}превысит" + str(CRIT_POOL_USAGE_PCT) + "% если случится -1ДЦ{{/isWarn}}{{/isOk}}",
            'url': link_sensor.build_sensor_link(parameters=dict(stack='false')),
        },
        description="JobsApp: утилизация тред-пула выполнения задач (окружение {env}, %)".format(env=env),
        type=Type(
            expression=Expression(
                program='''
                    let usage_pct = 100 * {running} / {size};
                    alarm_if(last(usage_pct) > {crit});

                    warn_if(last(usage_pct) * {dc} / ({dc} - 1) > {crit});
                '''.format(
                    running=running.selectors,
                    size=size.selectors,
                    dc=DCS_COUNT,
                    crit=CRIT_POOL_USAGE_PCT,
                ),
            ),
        ),
        notification_channels={
            direct_juggler_annotated.id,
        },
        window_secs=15 * 60,
        delay_seconds=0,
    )

prod_pool_alert = create_pool_alert(env='production')
test_pool_alert = create_pool_alert(env='testing')
devtest_pool_alert = create_pool_alert(env='devtest')
dev7_pool_alert = create_pool_alert(env='dev7')


prod_pool_check = Check(
    namespace=NAMESPACE_PROD,
    host='direct.disaster_alerts',
    service="hourglass_thread_pool_usage",
    aggregator="logic_or",
    ttl=3600,
    refresh_time=60,
    flaps_config=FlapOptions(stable=600, critical=2700),
    children=[
        Child(host=get_solomon_host(HOST_SUFFIX), service=get_service('production')),
    ],
    tags=get_warden_tags(slug='direct_background-services_jobs_app_health'),
    meta={
        "urls": [
            {
                "type": "yasm_alert",
                "title": "график утилизации",
                "url": prod_pool_alert.annotations['url']
            }, {
                "type": "wiki",
                "title": "документация на алерт",
                "url": "https://docs.yandex-team.ru/direct-dev/reference/alerts/jobs-thread-pool-usage"
            },
        ],
    },
    notifications=[
        JUGGLER_HISTORY,
    ],
)


exports=[
    prod_pool_alert,
    test_pool_alert,
    devtest_pool_alert,
    dev7_pool_alert,
    prod_pool_check,
]
