from juggler_sdk import Check, Child
from library.python.monitoring.solo.objects.solomon.v2 import MultiAlert, Type, Expression
from library.python.monitoring.solo.objects.solomon.sensor import Sensor

from direct.solo.registered.channel import channels
from direct.solo.registered.juggler import reference
from direct.solo.registered.project import projects

MDS_SERVICES_WITH_AVATARS = [
    'advq',
    'BannerStorage',
    'canvas',
    'direct',
    'indoor',
]

MDS_SERVICES_WITH_STORAGE = [
    'BannerStorage',
    'canvas',
    'direct',
    'gorynych',
]

MDS_SERVICES_WITH_S3 = [
    'advq',
    'BannerStorage',
    'canvas',
    'direct',
    'directadmin',
    'DIRECT_AUTOTESTS',
    'directclient',
    'direct-dna',
    'dooh',
    'indoor',
]


def get_mds_usage_alert(id, abc_services, type):
    if not abc_services:
        raise Exception('empty ABC services list provided')
    def create_sensor(attribute):
        return Sensor(
            project="dispenser_common_prod", service="dispenser_db_prod", attribute=attribute,
            cluster="dispenser_qloud_env", host="dispenser", type="STORAGE",
            segment="-",  # иначе приходят паразитные значения https://st.yandex-team.ru/DIRECTADMIN-9659#61cc15f419a93c53a94ed176
            **{'service-name': type, 'project-key': '|'.join(abc_services)}
        )
    alert = MultiAlert(
        id=id,
        project_id=projects.direct.id,
        name='Использование квоты MDS {} ABC-сервисом (Solo)'.format(type),
        window_secs=60*60,
        delay_seconds=0,
        group_by_labels={'project-key'},
        annotations={
            'jugglerHostSuffix': id,
            'jugglerService': 'abc_{{labels.project-key}}',
            'description': 'Потребление дисковой квоты ABC-сервисом {{labels.project-key}} (https://dispenser.yandex-team.ru/db/projects/{{labels.project-key}})',
            'url': 'https://solomon.yandex-team.ru/admin/projects/direct/alerts/{{alert.id}}',
        },
        type=Type(
            expression=Expression(
                program="""
                let actual = {actual};
                let max = {max};

                let usage = actual / max;
                let usage_pct = avg(usage) * 100;
                let usage_pct_str = to_fixed(usage_pct, 2);

                let crit = 90;
                let warn = 80;
                let trafficColor = usage_pct > crit ? "red" : (usage_pct > warn ? "yellow" : "green");
                warn_if(usage_pct > warn);
                alarm_if(usage_pct > crit);
                """.format(
                    actual=create_sensor('actual'),
                    max=create_sensor('max'),
                ),
            ),
        ),
        notification_channels={channels.direct_juggler_annotated.id},
        resolved_empty_policy="RESOLVED_EMPTY_NO_DATA",
    )
    return alert


def get_all_alerts():
    exports = []
    children_for_dashboard = []
    for item in [
        ('mds-usage-avatars', MDS_SERVICES_WITH_AVATARS, 'Avatars'),
        ('mds-usage-mds', MDS_SERVICES_WITH_STORAGE, 'MDS'),
        ('mds-usage-s3', MDS_SERVICES_WITH_S3, 'S3'),
    ]:
        alert = get_mds_usage_alert(*item)
        alert_id, abc_services, type = item
        aggregate = Check(
            namespace=reference.NAMESPACE_PROD,
            host='direct.prod_resources.mds',
            service=type.lower(),
            refresh_time=90, ttl=900,
            aggregator='logic_or',
            children=[
                Child(
                    host='direct.solomon-alert.'+alert.annotations['jugglerHostSuffix'],    # дублирование логики, описанной в канале алерта
                    service='abc_'+abc_service_slug     # см. аннотации алерта выше
                )
                for abc_service_slug in abc_services],
            meta={
                "urls": [
                    {
                        "type": "wiki",
                        "title": "Документация",
                        "url": "https://docs.yandex-team.ru/direct-dev/reference/alerts/prod_resources_mds",
                    }
                ]
            },
        )
        children_for_dashboard.append(Child(host='direct.prod_resources.mds', service=type.lower()))
        exports.append(alert)
        exports.append(aggregate)
    exports.append(
        Check(
            namespace=reference.NAMESPACE_PROD,
            host='direct.prod_resources',
            service='mds',
            tags=['direct-resources'],
            refresh_time=90, ttl=900,
            aggregator='logic_or',
            children=children_for_dashboard,
            meta={
                "urls": [
                    {
                        "type": "wiki",
                        "title": "Документация",
                        "url": "https://docs.yandex-team.ru/direct-dev/reference/alerts/prod_resources_mds",
                    }
                ]
            },
        )
    )
    return exports
