from juggler_sdk import Check, Child, FlapOptions
from library.python.monitoring.solo.objects.solomon.v2 import Alert, Type, Expression

from direct.solo.registered.channel import channels
from direct.solo.registered.dashboard.bs_export import bsexport_common
from direct.solo.registered.graph.bs_export import all_queue_delayed_clients_percent, std_graph, buggy_graph
from direct.solo.registered.juggler import reference
from direct.solo.registered.juggler.reference import get_warden_tags
from direct.solo.registered.project import projects
from direct.solo.registered.sensors.external_metrics import direct_banners_log_requests
from direct.solo.registered.sensors.transport_queues import bs_export_campaigns_count, bs_export_buggy_sequence_age
from direct.solo.registered.sensors.transport_queues import bs_export_clients_count, bs_export_max_queue_age, BS_EXPORT_CRIT_DELAY

LB_ERRORS_WINDOW = 3
BSSOAP_WINDOW = 10
BSSOAP_UNDONE_WARN = 2000
BSSOAP_UNDONE_CRIT = 15000
BSSOAP_WARN = 3
LB_SIZE_WINDOW = 10
STD_SLA_CLIENTS_WARN=2
STD_SLA_CLIENTS_CRIT=20
BUGGY_CAMPAIGNS_COUNT_WARN=400
BUGGY_CAMPAIGNS_COUNT_CRIT=2000
HOST='direct.perl_bsexport'
HOST_SUFFIX = "bsexport"
SERVICE_LOGBROKER_WRITE_FAILS = "logbroker_write_fails"
SERVICE_LOGBROKER_MESSAGE_SIZE_PROBLEMS = "logbroker_message_size_problems"
SERVICE_STD_DELAYED_CLIENTS = 'queue_std.delayed_clients_count'
SERVICE_STD_QUEUE_P80_AGE = 'queue_std.age_minutes_p80'
SERVICE_STD_QUEUE_P100_AGE = 'queue_std.age_minutes_p100'
SERVICE_BUGGY_CAMPAIGNS_COUNT = 'queue_buggy.campaigns_count'


lb_errors_sensor = direct_banners_log_requests.mutate(interpreted_status='server_error')
logbroker_write_fails = Alert(
    id='bsexport_' + SERVICE_LOGBROKER_WRITE_FAILS,
    project_id=projects.direct.id,
    name='Ошибки при отправке в БК через logboker',
    description='Проверяем количество ошибок при отправке в logbroker из bsClientData',
    annotations={
        "jugglerHostSuffix": HOST_SUFFIX,
        "jugglerService": SERVICE_LOGBROKER_WRITE_FAILS,
        "jugglerDescription": "Ошибок отправки в logbroker за последние " + str(LB_ERRORS_WINDOW) + " минуты: {{expression.errors}}",
    },
    window_secs=LB_ERRORS_WINDOW*60,
    type=Type(
        expression=Expression(
            program="""
                let error_rate = {errors};
                let errors = last(integral(error_rate));
                let border = last(integral({success})) * 0.05;

                alarm_if(errors > border);
                warn_if(errors > {warn});
            """.format(
                errors=lb_errors_sensor.selectors,
                success=direct_banners_log_requests.mutate(interpreted_status='success').selectors,
                warn=LB_ERRORS_WINDOW,
            )
        ),
    ),
    notification_channels={
        channels.direct_juggler_annotated.id,
    },
    resolved_empty_policy="RESOLVED_EMPTY_NO_DATA",
)

lb_long_msg_sensor = direct_banners_log_requests.mutate(interpreted_status='client_error')
logbroker_message_size_problem = Alert(
    id='bsexport_' + SERVICE_LOGBROKER_MESSAGE_SIZE_PROBLEMS,
    project_id=projects.direct.id,
    name='Превышение размера сообщения для отправки в БК через logbroker',
    description='Проверяем количество ошибок, когда сформированное к отправке в БК через logbroker сообщение превысило допустимый размер',
    annotations={
        "jugglerHostSuffix": HOST_SUFFIX,
        "jugglerService": SERVICE_LOGBROKER_MESSAGE_SIZE_PROBLEMS,
        "jugglerDescription": "{{expression.errors}} попыток за последние "
            + str(LB_SIZE_WINDOW) + " минут отправить в logbroker сообщение, превышающее допустимый размер",
    },
    window_secs=LB_SIZE_WINDOW*60,
    type=Type(
        expression=Expression(
            program="""
                let error_rate = {errors};
                let errors = last(integral(error_rate));

                alarm_if(errors > 1);
            """.format(
                errors=lb_long_msg_sensor.selectors,
            ),
        ),
    ),
    notification_channels={
        channels.direct_juggler_annotated.id,
    },
    resolved_empty_policy="RESOLVED_EMPTY_NO_DATA",
)
logbroker_requests_fails_check = Check(
    namespace=reference.NAMESPACE_PROD,
    host=HOST,
    service="logbroker_request_fails",
    aggregator="logic_or",
    ttl=600,
    refresh_time=60,
    children=[
        Child(host=channels.get_solomon_host(HOST_SUFFIX), service=SERVICE_LOGBROKER_WRITE_FAILS),
        Child(host=channels.get_solomon_host(HOST_SUFFIX), service=SERVICE_LOGBROKER_MESSAGE_SIZE_PROBLEMS),
    ],
    meta={
        "urls": [
            {
                "type": "yasm_alert",
                "title": "rate ошибок",
                "url": direct_banners_log_requests.mutate(interpreted_status=('!=', 'success')).build_sensor_link({'b': '3h', 'stack': 'false'}),
            }, {
                "type": "wiki",
                "title": "документация",
                "url": "https://docs.yandex-team.ru/direct-dev/reference/alerts/bs-export-logbroker-request-fails",
            }
        ]
    },
    tags=get_warden_tags(slug='direct_bsexport_logbroker_fails'),
    notifications=[
        reference.JUGGLER_HISTORY,
        reference.CALLING_CHECKS_CHAT,
        reference.DUTY_ESCALATION,
    ],
)

std_sla = Alert(
    id='bsexport_' + SERVICE_STD_DELAYED_CLIENTS,
    project_id=projects.direct.id,
    name='Задержки в экспорте в БК очередью std',
    description='Клиентов в std-очереди экспорта с временем более {border} минут'.format(border=BS_EXPORT_CRIT_DELAY),
    annotations={
        "jugglerHostSuffix": HOST_SUFFIX,
        "jugglerService": SERVICE_STD_DELAYED_CLIENTS,
        "jugglerDescription": "Клиентов в очереди std с возрастом более {border} минут: {count} (пороги: warn {warn} / crit {crit})".format(
            count='{{expression.count}}',
            border=BS_EXPORT_CRIT_DELAY,
            warn=STD_SLA_CLIENTS_WARN,
            crit=STD_SLA_CLIENTS_CRIT,
        ),
    },
    window_secs=BSSOAP_WINDOW*60,
    type=Type(
        expression=Expression(
            program="""
                let delayed_clients = histogram_count({border}, inf(), {sensor});
                let count = last(delayed_clients);

                alarm_if(count > {crit});
                warn_if(count > {warn});
            """.format(
                border=BS_EXPORT_CRIT_DELAY,  # должно попадать на существующий шаг корзин в сенсоре
                sensor=bs_export_clients_count.mutate(queue='std').selectors,
                warn=STD_SLA_CLIENTS_WARN,
                crit=STD_SLA_CLIENTS_CRIT,
            )
        ),
    ),
    notification_channels={
        channels.direct_juggler_annotated.id,
    },
    resolved_empty_policy="RESOLVED_EMPTY_NO_DATA",
)
std_sla_check = Check(
    namespace=reference.NAMESPACE_PROD,
    host=HOST,
    service=SERVICE_STD_DELAYED_CLIENTS,
    aggregator="logic_or",
    ttl=600,
    refresh_time=60,
    flaps_config=FlapOptions(stable=210, critical=900),
    children=[
        Child(host=channels.get_solomon_host(HOST_SUFFIX), service=SERVICE_STD_DELAYED_CLIENTS),
    ],
    tags=get_warden_tags(slug='direct_bsexport_delayed_clients'),
    meta={
        "urls": [
            {
                "type": "yasm_alert",
                "title": "Доля клиентов с отставанием",
                "url": all_queue_delayed_clients_percent.get_full_link('3h') + '&queue=std',
            }, {
                "type": "wiki",
                "title": "документация",
                "url": "https://docs.yandex-team.ru/direct-dev/reference/alerts/bs-export-delayed-clients-std",
            }
        ]
    },
    notifications=[
        reference.JUGGLER_HISTORY,
        reference.CALLING_CHECKS_CHAT,
        reference.DUTY_ESCALATION,
    ],
)


def std_age_alert(percentile, warn, crit):
    sensor = bs_export_max_queue_age.mutate(queue='std', percentile=percentile)
    return Alert(
        id='bsexport_queue_std.age_minutes_' + percentile,
        project_id=projects.direct.id,
        name=percentile + ' возраста кампаний в std-очереди экспорта в БК',
        description='',
        annotations={
            "jugglerHostSuffix": HOST_SUFFIX,
            "jugglerService": 'queue_std.age_minutes_' + percentile,
            "url": "https://solomon.yandex-team.ru/admin/projects/direct/alerts/{{alert.id}}\n" +
                sensor.build_sensor_link({'b': '3h', 'stack': 'false'}),
            "jugglerDescription": "{pct} возраста очереди std: {age} минут (пороги: warn {warn} / crit {crit})".format(
                pct=percentile,
                age='{{expression.age}}',
                warn=warn,
                crit=crit,
            ),
        },
        window_secs=BSSOAP_WINDOW*60,
        type=Type(
            expression=Expression(
                program="""
                    let pct = {sensor};
                    let age = round(last(pct));

                    alarm_if(last(pct) > {crit});
                    warn_if(last(pct) > {warn});
                """.format(
                    sensor=sensor.selectors,
                    warn=warn,
                    crit=crit,
                ),
            ),
        ),
        notification_channels={
            channels.direct_juggler_annotated.id,
        },
        resolved_empty_policy="RESOLVED_EMPTY_NO_DATA",
    )


queue_age_p80_std = std_age_alert(percentile='p80', warn=99999, crit=BS_EXPORT_CRIT_DELAY/2)    # 99999 как заглушка
queue_age_p99_std = std_age_alert(percentile='p99', warn=BS_EXPORT_CRIT_DELAY, crit=BS_EXPORT_CRIT_DELAY*2)
queue_age_p100_std = std_age_alert(percentile='p100', warn=BS_EXPORT_CRIT_DELAY*3, crit=1440)

std_queue_age_check = Check(
    namespace=reference.NAMESPACE_PROD,
    host=HOST,
    service='queue_std.age_minutes',
    aggregator="logic_or",
    ttl=600,
    refresh_time=60,
    flaps_config=FlapOptions(stable=300, critical=1200),
    children=[
        Child(host=channels.get_solomon_host(HOST_SUFFIX), service=queue_age_p80_std.annotations['jugglerService']),
        Child(host=channels.get_solomon_host(HOST_SUFFIX), service=queue_age_p99_std.annotations['jugglerService']),
        Child(host=channels.get_solomon_host(HOST_SUFFIX), service=queue_age_p100_std.annotations['jugglerService']),
    ],
    meta={
        "urls": [
            {
                "type": "yasm_alert",
                "title": "состояние очереди std",
                "url": std_graph.get_full_link('3h'),
            }, {
                "type": "wiki",
                "title": "документация",
                "url": "https://docs.yandex-team.ru/direct-dev/reference/alerts/bs-export-queue-age-std",
            }
        ]
    },
    notifications=[
        reference.JUGGLER_HISTORY,
    ],
)


def other_queue_alert(queue, warn_bin, warn_border=1):
    name = 'queue_' + queue + '.age_minutes'
    return Alert(
        id='bsexport_' + name,
        project_id=projects.direct.id,
        name='Наличие кампаний с большим возрастом в очереди экспорта БК ' + queue,
        description='',
        annotations={
            "jugglerHostSuffix": HOST_SUFFIX,
            "jugglerService": name,
            "jugglerDescription": 'В очереди ' + queue + ' {{expression.count}} кампаний с возрастом больше ' + str(warn_bin) + ' минут'
                + '{{#isAlarm}}, из них у {{expression.crit_count}} возраст превышает сутки{{/isAlarm}}'
        },
        window_secs=BSSOAP_WINDOW*60,
        type=Type(
            expression=Expression(
                program="""
                    let campaigns_count = histogram_count({warn_bin}, inf(), {sensor});
                    let count = last(campaigns_count);

                    let crit_count = last(histogram_count({crit_bin}, inf(), {sensor}));

                    alarm_if(crit_count > 0);
                    warn_if(count > 0);
                """.format(
                    sensor=bs_export_campaigns_count.mutate(queue=queue).selectors,
                    warn_bin=warn_bin,
                    crit_bin=1440,  # максимальная корзина
                )
            ),
        ),
        notification_channels={
            channels.direct_juggler_annotated.id,
        },
        resolved_empty_policy="RESOLVED_EMPTY_NO_DATA",
    )

internal_ads_queue_age = other_queue_alert(queue='internal_ads', warn_bin=45)
buggy_queue_age = other_queue_alert(queue='buggy', warn_bin=240)
fast_queue_age = other_queue_alert(queue='fast', warn_bin=90)
preprod_queue_age = other_queue_alert(queue='preprod', warn_bin=60)
other_queue_age = other_queue_alert(queue='other', warn_bin=180)

queue_age_heavy = Alert(
    id='bsexport_queue_heavy.age_minutes',
    project_id=projects.direct.id,
    name='Возраст кампаний в heavy-очереди экспорта в БК',
    description='',
    annotations={
        "jugglerHostSuffix": HOST_SUFFIX,
        "jugglerService": 'queue_heavy.age_minutes',
        "url": "https://solomon.yandex-team.ru/admin/projects/direct/alerts/{{alert.id}}\n" +
            bs_export_max_queue_age.mutate(queue='heavy', percentile='p80|p100').build_sensor_link({'b': '3h', 'stack': 'false'}),
        "jugglerDescription": "возраст очереди heavy: p80 {{expression.p80}}, p100 {{expression.p100}} минут",
    },
    window_secs=BSSOAP_WINDOW*60,
    type=Type(
        expression=Expression(
            program="""
                let age_p100 = {p100_sensor};
                let age_p80 = {p80_sensor};

                let p100 = round(last(age_p100));
                let p80 = round(last(age_p80));

                alarm_if(p100 > {crit});
                warn_if(p80 > {warn});
            """.format(
                p80_sensor=bs_export_max_queue_age.mutate(queue='heavy', percentile='p80').selectors,
                p100_sensor=bs_export_max_queue_age.mutate(queue='heavy', percentile='p100').selectors,
                warn=60*10,
                crit=60*24*2,
            ),
        ),
    ),
    notification_channels={
        channels.direct_juggler_annotated.id,
    },
    resolved_empty_policy="RESOLVED_EMPTY_NO_DATA",
)

other_queues_age_check=Check(
    namespace=reference.NAMESPACE_PROD,
    host=HOST,
    service="other_queues_age",
    aggregator="logic_or",
    ttl=600,
    refresh_time=60,
    flaps_config=FlapOptions(stable=600, critical=1800),
    children=[
        Child(host=channels.get_solomon_host(HOST_SUFFIX), service=internal_ads_queue_age.annotations['jugglerService']),
        Child(host=channels.get_solomon_host(HOST_SUFFIX), service=buggy_queue_age.annotations['jugglerService']),
        Child(host=channels.get_solomon_host(HOST_SUFFIX), service=fast_queue_age.annotations['jugglerService']),
        Child(host=channels.get_solomon_host(HOST_SUFFIX), service=preprod_queue_age.annotations['jugglerService']),
        Child(host=channels.get_solomon_host(HOST_SUFFIX), service=other_queue_age.annotations['jugglerService']),
        Child(host=channels.get_solomon_host(HOST_SUFFIX), service=queue_age_heavy.annotations['jugglerService']),
    ],
    meta={
        "urls": [
            {
                "type": "yasm_alert",
                "title": "Дашборд про экспорт",
                "url": bsexport_common.get_link() + '&b=6h',
            }, {
                "type": "wiki",
                "title": "документация",
                "url": "https://docs.yandex-team.ru/direct-dev/reference/alerts/bs-export-other-queues-age",
            }
        ]
    },
    notifications=[
        reference.JUGGLER_HISTORY,
    ],
)

sequence_age_buggy = Alert(
    id='bsexport_queue_buggy.age_by_order',
    project_id=projects.direct.id,
    name='Проверка ротации кампаний в buggy очереди экспорта в БК',
    description='Проверяем, что кампании переобходятся по кругу, а не заблокировали собой очередь',
    annotations={
        "jugglerHostSuffix": HOST_SUFFIX,
        "jugglerService": 'queue_buggy.age_by_order',
        "url": bs_export_buggy_sequence_age.build_sensor_link({'b': '3h', 'stack': 'false'}),
        "jugglerDescription": "возраст по порядку обработки в очереди buggy: {{expression.age}} минут",
    },
    window_secs=BSSOAP_WINDOW*60,
    type=Type(
        expression=Expression(
            program="""
                let sequence_age = {sensor};
                let age = last(sequence_age);

                alarm_if(age > {crit});
                warn_if(age > {warn});
            """.format(
                sensor=bs_export_buggy_sequence_age.selectors,
                warn=60,
                crit=2*60,
            ),
        ),
    ),
    notification_channels=set(),
    resolved_empty_policy="RESOLVED_EMPTY_NO_DATA",
)

buggy_campaigns_sensor = bs_export_campaigns_count.mutate(queue='buggy')
buggy_campaigns_count = Alert(
    id='bsexport_' + SERVICE_BUGGY_CAMPAIGNS_COUNT,
    project_id=projects.direct.id,
    name='Проверяем, что в buggy-очереди очереди экспорта в БК не очень много кампаний',
    description='',
    annotations={
        "jugglerHostSuffix": HOST_SUFFIX,
        "jugglerService": SERVICE_BUGGY_CAMPAIGNS_COUNT,
        "url": buggy_campaigns_sensor.build_sensor_link({'b': '3h', 'cs': 'gradient', 'stack': 'true'}),
        "jugglerDescription": "Количество кампаний в очереди buggy: {count} (пороги: warn {warn} / crit {crit})".format(
            count='{{expression.count}}',
            warn=BUGGY_CAMPAIGNS_COUNT_WARN,
            crit=BUGGY_CAMPAIGNS_COUNT_CRIT,
        ),
    },
    window_secs=BSSOAP_WINDOW*60,
    type=Type(
        expression=Expression(
            program="""
                let campaigns_count = histogram_count({sensor});
                let count = last(campaigns_count);

                alarm_if(count > {crit});
                warn_if(count > {warn});
            """.format(
                sensor=buggy_campaigns_sensor.selectors,
                warn=BUGGY_CAMPAIGNS_COUNT_WARN,
                crit=BUGGY_CAMPAIGNS_COUNT_CRIT,
            ),
        ),
    ),
    notification_channels={
        channels.direct_juggler_annotated.id,
    },
    resolved_empty_policy="RESOLVED_EMPTY_NO_DATA",
)

buggy_campaigns_count_check = Check(
    namespace=reference.NAMESPACE_PROD,
    host=HOST,
    service=SERVICE_BUGGY_CAMPAIGNS_COUNT,
    aggregator="logic_or",
    ttl=600,
    refresh_time=60,
    flaps_config=FlapOptions(stable=1200, critical=2700),
    children=[
        Child(host=channels.get_solomon_host(HOST_SUFFIX), service=SERVICE_BUGGY_CAMPAIGNS_COUNT),
    ],
    meta={
        "urls": [
            {
                "type": "yasm_alert",
                "title": "состояние buggy-очереди",
                "url": buggy_graph.get_full_link('3h'),
            }, {
                "type": "wiki",
                "title": "документация",
                "url": "https://docs.yandex-team.ru/direct-dev/reference/alerts/bs-export-buggy-campaigns-count",
            }
        ]
    },
    notifications=[
        reference.JUGGLER_HISTORY,
    ],
)

exports=[
    logbroker_write_fails,
    logbroker_message_size_problem,
    logbroker_requests_fails_check,
    std_sla,
    std_sla_check,
    queue_age_p80_std,
    queue_age_p99_std,
    queue_age_p100_std,
    std_queue_age_check,
    internal_ads_queue_age,
    buggy_queue_age,
    fast_queue_age,
    preprod_queue_age,
    other_queue_age,
    queue_age_heavy,
    other_queues_age_check,
    sequence_age_buggy,
    buggy_campaigns_count,
    buggy_campaigns_count_check,
]
