from library.python.monitoring.solo.objects.solomon.v2 import Dashboard, Row, Panel, Parameter

from direct.solo.registered.alert.infra.mdb.mysql.common import RUNNING_THREADS_CRIT_THRESHOLD, REPLICATION_LAG_THRESHOLD
from direct.solo.registered.cluster import mysql
from direct.solo.registered.graph.mdb import net_graph, cpu_graph, disk_free_graph, disk_io_graph, mysql_threads_connected_graph, \
    ppcdata_master_cpu_usage_graph, ppcdata_master_queries_graph, ppcdata_master_threads_graph, ppcdata_master_threads_connected_graph, \
    ppcdata_master_slow_queries_graph, ppcdata_master_avg_query_time_graph, \
    ppcdata_mdb_disk_usage_mysql_data_graph, ppcdata_master_binlog_age_graph
from direct.solo.registered.project import projects
from direct.solo.registered.sensors.mdb import MdbByHostSensor, mysql_threads_running, mysql_replication_lag


def generate_links(cluster_mapping, dashboard_url, sandbox_id):
    parts = []
    mapping = list(cluster_mapping.items())
    if sandbox_id is not None:
        mapping.append(('sandbox', sandbox_id))
    for (name, cluster_id) in mapping:
        dash_url = dashboard_url + '&cluster=mdb_' + cluster_id + '&instance=' + name
        dash_url += '&b={{b}}&e={{e}}&graphOnly={{graphOnly}}'
        yc_url = 'https://yc.yandex-team.ru/folders/fooa07bcrr7souccreru/managed-mysql/cluster/' + cluster_id
        links_str = '{name}: [dash]({dash_url}), [YC]({yc_url})'.format(name=name, dash_url=dash_url, yc_url=yc_url)
        parts.append(links_str)
    links = "  \n".join(parts)
    return links + "\n\nтекущий деш — {{instance}}"


def get_links_panel(cluster_mapping, dashboard, sandbox_id=None):
    return Panel(
        type='MARKDOWN',
        title='Ссылки',
        url='',
        markdown=generate_links(cluster_mapping, dashboard.get_link(), sandbox_id),
        colspan=1,
        rowspan=4,
    )


def get_node_graph(graph, node):
    return graph.get_dashboard_link(
        parametrise={
            'cluster': '{{cluster}}',
            'dc': 'by_node',
            'host': 'by_node',
            'node': node
        },
        use_legend=False,
    )


def get_net_graph(node):
    return get_node_graph(net_graph, node)


def get_cpu_graph():
    return get_node_graph(cpu_graph, 'primary')


def get_disk_io_graph():
    return get_node_graph(disk_io_graph, 'primary')


def get_free_space_graph():
    return disk_free_graph.get_dashboard_link(
        parametrise={'cluster': '{{cluster}}'},
        use_legend=False,
    )


def get_connections_graph():
    return mysql_threads_connected_graph.get_dashboard_link(
        parametrise={'cluster': '{{cluster}}'},
        use_legend=False,
    )


is_alive = MdbByHostSensor(sensor='mysql_is_alive')
is_primary = MdbByHostSensor(sensor='mysql_is_primary')
queries = MdbByHostSensor(sensor='mysql_Queries_rate')
slow_queries = MdbByHostSensor(sensor='mysql_Slow_queries_rate')

prod_mysql = Dashboard(
    id='mdb-mysql',
    project_id=projects.direct.id,
    name='MDB MySQL {{instance}}',
    parameters={
        Parameter(name='cluster', value='*'),
        Parameter(name='instance', value='*'),
        Parameter(name='b', value='6h'),
        Parameter(name='e', value='*'),
        Parameter(name='graphOnly', value='y'),
    },
    rows=[],
    height_multiplier=1.0,
)

prod_mysql.rows=[
    Row(panels=[
        Panel(title='Connections', url=get_connections_graph(), colspan=2, rowspan=1),
        Panel(title="CPU", subtitle='primary',  url=get_cpu_graph(), colspan=2, rowspan=1),
        Panel(title="Queries", url=queries.get_dashboard_link(), colspan=2, rowspan=1),
        get_links_panel(mysql.production, prod_mysql, mysql.sandbox),
    ]),
    Row(panels=[
        Panel(title='Running threads', url=mysql_threads_running.get_dashboard_link(threshold=str(RUNNING_THREADS_CRIT_THRESHOLD)), colspan=2, rowspan=1),
        Panel(title="Replication lag", url=mysql_replication_lag.get_dashboard_link(threshold=str(REPLICATION_LAG_THRESHOLD)), colspan=2, rowspan=1),
        Panel(title="Slow queries", url=slow_queries.get_dashboard_link(), colspan=2, rowspan=1),
    ]),
    Row(panels=[
        Panel(title="Is Alive", url=is_alive.get_dashboard_link(interpolate='none', stack='true'), colspan=2, rowspan=1),
        Panel(title="Network bytes", subtitle='primary', url=get_net_graph('primary'), colspan=2, rowspan=1),
        Panel(title="Free space, bytes", url=get_free_space_graph(), colspan=2, rowspan=1),
    ]),
    Row(panels=[
        Panel(title="Is Primary", url=is_primary.get_dashboard_link(interpolate='none', stack='true'), colspan=2, rowspan=1),
        Panel(title="Network bytes", subtitle='replica', url=get_net_graph('replica'), colspan=2, rowspan=1),
        Panel(title="Disk IO, bytes", subtitle="primary", url=get_disk_io_graph(), colspan=2, rowspan=1),
    ]),
    Row(panels=[
        Panel(
            title='juggler',
            url='https://juggler.yandex-team.ru/aggregate_checks/?sort=SERVICES_ASC&query=namespace=direct.prod%26tag=mysql.{{instance}}&iframe=true',
            colspan=3,
            rowspan=2,
        ),
    ])
]

prod_mysql_ppcdata_masters = Dashboard(
    id='mdb-ppcdata-masters',
    project_id=projects.direct.id,
    name='MDB MySQL shard masters',
    rows=[
        Row(panels=[
            Panel(title='CPU usage %', url=ppcdata_master_cpu_usage_graph.get_dashboard_link(), colspan=2, rowspan=1),
            Panel(title='Average query time', url=ppcdata_master_avg_query_time_graph.get_dashboard_link(), colspan=2, rowspan=1),
        ]),
        Row(panels=[
            Panel(title='Threads', url=ppcdata_master_threads_connected_graph.get_dashboard_link(), colspan=2, rowspan=1),
            Panel(title='Connections', url=ppcdata_master_threads_graph.get_dashboard_link(), colspan=2, rowspan=1),
        ]),
        Row(panels=[
            Panel(title='Queries', url=ppcdata_master_queries_graph.get_dashboard_link(), colspan=2, rowspan=1),
            Panel(title='Slow queries', url=ppcdata_master_slow_queries_graph.get_dashboard_link(), colspan=2, rowspan=1),
        ]),
        Row(panels=[
            Panel(title='Binlog age in days', url=ppcdata_master_binlog_age_graph.get_dashboard_link(), colspan=2, rowspan=1),
            Panel(
                title="Links",
                type="MARKDOWN",
                url="",
                markdown="""[MySQL data (disk usage)]({disk_usage_mysql_data_link})""".format(
                    disk_usage_mysql_data_link=ppcdata_mdb_disk_usage_mysql_data_graph.get_dashboard_link() + "&b=31d",
                ),
            ),
        ]),
    ],
)


def mysql_cpu_graph_panel(name, cluster_id):
    url = cpu_graph.get_dashboard_link(parametrise={
        'cluster': 'mdb_' + cluster_id,
        'dc': 'by_node',
        'host': 'by_node',
        'node': 'primary'
    })
    return Panel(title="CPU " + name, url=url)


def prod_mysql_cpu_rows(cluster_mapping):
    cols = 4
    rows = []
    clusters = list(cluster_mapping.items())
    for start in range(0, len(clusters), cols):
        rows.append(Row(panels=[mysql_cpu_graph_panel(*x) for x in clusters[start:start+cols]]))
    return rows


prod_mysql_cpu = Dashboard(
    id='mdb-mysql_cpu',
    project_id=projects.direct.id,
    name='MDB MySQL CPU',
    rows=prod_mysql_cpu_rows(mysql.production),
    height_multiplier=1.0,
)


def generate_links_short(cluster_mapping, dashboard_url, sandbox_id):
    parts = []
    mapping = list(cluster_mapping.items())
    if sandbox_id is not None:
        mapping.append(('sandbox', sandbox_id))
    for (name, cluster_id) in mapping:
        dash_url = dashboard_url + '&cluster=mdb_' + cluster_id + '&instance=' + name + '&b=6h'
        links_str = '[{name}]({dash_url})'.format(name=name, dash_url=dash_url)
        parts.append(links_str)
    links = "\n".join(parts)
    links += "\n\n" + "[CPU для всех шардов]({url})".format(url=prod_mysql_cpu.get_link())
    return links


prod_mysq_links = Dashboard(
    id='mdb-mysql-links',
    project_id=projects.direct.id,
    name="Ссылки на дашборды MySQL",
    parameters=set(),
    height_multiplier=1.0,
    rows=[
        Row(panels=[
            Panel(title="", type="MARKDOWN", markdown=generate_links_short(mysql.production, prod_mysql.get_link(), mysql.sandbox), url="")
        ]),
    ],
)


exports=[
    prod_mysql,
    prod_mysq_links,
    prod_mysql_cpu,
    prod_mysql_ppcdata_masters,
]
