package ru.yandex.direct.web;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.function.BiFunction;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.operation.Operation;
import ru.yandex.direct.operation.execution.AllOrNothingExecutionStrategy;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.validation.model.WebValidationResult;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;

/**
 * Аггрегатор нескольких операций.
 * <p>
 * Умеет формировать суммарный результат валидации уровня веба,
 * содержащий результаты валидации всех входящих в него операций.
 * <p>
 * Операции выполняет по принципу "все или ничего". Сначала проводит
 * все подготовительные шаги для всех операций ({@link Operation#prepare()},
 * которые включают валидацию. Если хотябы одна операция провалилась
 * на подготовительном этапе, то формирует суммарный результат валидации всех операций,
 * который содержит результаты валидации только тех операций, которые провалились.
 * Если подготовка у всех операций прошла успешно, то выполняет все операции.
 * <p>
 * Пример.
 * <p>
 * Контроллер работает сразу с кампаниями и группами объявлений.
 * Запрос имеет такой вид:
 * <pre> {@code
 * {
 *     "campaigns": [...],
 *     "adGroups": [...]
 * }
 * }</pre>
 * <p>
 * Для этого запроса создаются две операции. Эти операции передаются в агрегатор с именами
 * "campaigns" и "adGroups".
 * <p>
 * Если хотя бы одна из операций провалится на этапе валидации,
 * то будет возвращен суммарный результат валидации, в котором будут присутствовать
 * результаты валидации только тех операций, которые не прошли валидацию:
 * <pre>
 * <code>
 * {
 *     "fields": {
 *         "campaigns": {
 *             // отрицательный результат валидации по операции над кампаниями или null
 *         },
 *         "adGroups": {
 *             // отрицательный результат валидации по операции над группами объявлений или null
 *         }
 *     }
 * }
 * </code>
 * </pre>
 * <p>
 * Если же обе операции выполнятся успешно, то будет возвращен пустой {@link Optional}.
 */
@ParametersAreNonnullByDefault
public class WebOperationAggregatorWithNames {

    private final AllOrNothingExecutionStrategy allOrNothingExecutionStrategy;
    private final Map<String, ? extends Operation<?>> operations;
    private final BiFunction<ValidationResult<?, Defect>, Path, WebValidationResult>
            validationResultConverter;

    private WebOperationAggregatorWithNames(Map<String, ? extends Operation<?>> operations,
                                            BiFunction<ValidationResult<?, Defect>, Path, WebValidationResult> validationResultConverter) {
        this.validationResultConverter = validationResultConverter;
        this.allOrNothingExecutionStrategy = new AllOrNothingExecutionStrategy(true);
        this.operations = operations;
    }

    /**
     * Выполняет все операции по принципу "все или ничего". Подготовительный этап,
     * включая валидацию, выполняет у всех операций, даже если у некоторых этот этап
     * уже провалился. Это позволяет получить ошибки по всем операциям, а не только
     * по первой проваленной.
     *
     * @return если все операции успешно прошли валидацию и выполнились, то пустой Optional,
     * в противном случае суммарный результат валидации по всем операциям
     * (в суммарном результате валидации могут отсутствовать результаты по некоторым операциям,
     * если их валидация прошла успешно).
     */
    public Optional<WebValidationResult> executeAllOrNothing() {
        boolean successful = allOrNothingExecutionStrategy.execute(operations.values());
        return successful ?
                Optional.empty() :
                Optional.of(buildWebValidationResult());
    }

    private WebValidationResult buildWebValidationResult() {
        WebValidationResult webValidationResult = new WebValidationResult();
        operations.forEach((name, operation) -> {
            if (operation.getResult().isPresent()) {
                ValidationResult<?, Defect> vr = operation.getResult().get().getValidationResult();
                WebValidationResult subWebValidationResult =
                        validationResultConverter.apply(vr, path(field(name)));
                webValidationResult.addErrors(subWebValidationResult.getErrors());
                webValidationResult.addWarnings(subWebValidationResult.getWarnings());
            }
        });
        return webValidationResult;
    }

    @ParametersAreNonnullByDefault
    public static class Builder {
        private final Map<String, Operation<?>> operations = new HashMap<>();
        private BiFunction<ValidationResult<?, Defect>, Path, WebValidationResult> validationResultConverter;

        public Builder addOperation(String name, Operation operation) {
            operations.put(name, operation);
            return this;
        }

        public Builder addNullableOperation(String name, @Nullable Operation operation) {
            if (operation != null) {
                return addOperation(name, operation);
            }
            return this;
        }

        public Builder setValidationResultConverter(
                BiFunction<ValidationResult<?, Defect>, Path, WebValidationResult> validationResultConverter) {
            this.validationResultConverter = validationResultConverter;
            return this;
        }

        public WebOperationAggregatorWithNames build() {
            checkArgument(validationResultConverter != null, "validation result converter required");
            checkArgument(!operations.isEmpty(), "operations required");
            return new WebOperationAggregatorWithNames(operations, validationResultConverter);
        }
    }
}
