package ru.yandex.direct.web.admin;

import java.io.IOException;
import java.io.PrintWriter;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.google.common.net.HttpHeaders;
import com.google.common.net.MediaType;
import graphql.ExecutionResult;
import graphql.introspection.IntrospectionQuery;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;

import ru.yandex.direct.common.admin.engine.Action;
import ru.yandex.direct.common.admin.engine.ActionExecutor;
import ru.yandex.direct.grid.model.admin.GraphqlSchemaProvider;

import static graphql.ExecutionInput.newExecutionInput;
import static java.util.stream.Collectors.joining;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.JsonUtils.toJson;

/**
 * Админский контроллер - вывод graphql-схем без авторизации (для сборки фронтенда)
 */
@SuppressWarnings("unused")
@Action("graphql_schema")
public class GraphqlSchemaExecuter implements ActionExecutor {
    private static final Logger logger = LoggerFactory.getLogger(GraphqlSchemaExecuter.class);

    private final Map<String, GraphqlSchemaProvider> providersByName;

    public GraphqlSchemaExecuter(List<GraphqlSchemaProvider> providers) {
        this.providersByName = listToMap(providers, GraphqlSchemaProvider::getGraphqlApiName, Function.identity());
    }

    @Override
    public void execute(HttpServletRequest request, HttpServletResponse response) throws IOException {
        PrintWriter writer = response.getWriter();

        String schema = request.getParameter("schema");
        if (schema == null) {
            response.setStatus(HttpStatus.BAD_REQUEST.value());
            writer.write(errorMessage("Use parameter schema"));
            return;
        }
        if (!providersByName.containsKey(schema)) {
            response.setStatus(HttpStatus.NOT_FOUND.value());
            writer.write(errorMessage("Schema " + schema + " not found"));
            return;
        }

        ExecutionResult result = providersByName.get(schema)
                .getGraphql()
                .execute(newExecutionInput()
                        .query(IntrospectionQuery.INTROSPECTION_QUERY)
                        .variables(Collections.emptyMap())
                );
        response.setHeader(HttpHeaders.CONTENT_TYPE, MediaType.JSON_UTF_8.toString());
        writer.write(toJson(result));
    }

    private String errorMessage(String msg) {
        return msg + ". Available schemas: "
                + providersByName.keySet().stream()
                .sorted()
                .collect(joining(", "))
                + "\n";
    }
}
