package ru.yandex.direct.web.aspects;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.Arrays;

import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.web.bind.annotation.RequestBody;

import ru.yandex.direct.common.logging.RequestBodyLogWriter;
import ru.yandex.direct.web.logging.WebLogRecord;


@Aspect
@Component
public class RequestBodyLogAspect {

    private final RequestBodyLogWriter<WebLogRecord> requestBodyLogWriter;

    @Autowired
    public RequestBodyLogAspect(RequestBodyLogWriter<WebLogRecord> requestBodyLogWriter) {
        this.requestBodyLogWriter = requestBodyLogWriter;
    }

    @Pointcut("@within(org.springframework.stereotype.Controller)")
    public void controllerClass() {
    }

    @Pointcut("@annotation(org.springframework.web.bind.annotation.RequestMapping)")
    public void requestMappingMethod() {
    }

    @Pointcut("within(ru.yandex.direct.web..*)")
    public void webPackages() {
    }

    /**
     * Срабатывает на методах контроллеров (класс должен быть помечен аннотацией
     * {@link org.springframework.stereotype.Controller}), помеченных аннотацией
     * {@link org.springframework.web.bind.annotation.RequestMapping},
     * если они расположены в ru.yandex.direct.web или ниже.
     * Если у метода есть аргумент с аннотацией {@link org.springframework.web.bind.annotation.RequestBody},
     * добавляет этот объект в запись лога текущего запроса.
     */
    @Before("controllerClass() && requestMappingMethod() && webPackages()")
    public void logRequestBody(JoinPoint joinPoint) {

        int requestBodyIndex = getRequestBodyArgumentIndex(joinPoint);
        if (requestBodyIndex == -1) {
            return;
        }
        Object requestBodyArgument = joinPoint.getArgs()[requestBodyIndex];
        requestBodyLogWriter.writeBodyToLog(requestBodyArgument);
    }

    /**
     * Возвращает индекс аргумента с аннотацией RequestBody, если такой имеется или -1.
     */
    private int getRequestBodyArgumentIndex(JoinPoint joinPoint) {
        MethodSignature loggedMethodSignature = (MethodSignature) joinPoint.getSignature();
        Method loggedMethod = loggedMethodSignature.getMethod();
        Annotation[][] parameterAnnotations = loggedMethod.getParameterAnnotations();
        for (int i = 0; i < parameterAnnotations.length; ++i) {
            if (Arrays.stream(parameterAnnotations[i]).anyMatch(RequestBody.class::isInstance)) {
                return i;
            }
        }
        return -1;
    }
}
