package ru.yandex.direct.web.configuration;

import java.time.LocalDate;
import java.util.Arrays;
import java.util.List;

import javax.servlet.Filter;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.Module;
import com.fasterxml.jackson.databind.module.SimpleModule;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.FilterType;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Primary;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.web.filter.CharacterEncodingFilter;
import org.springframework.web.filter.CompositeFilter;
import org.springframework.web.servlet.LocaleResolver;
import org.springframework.web.servlet.config.annotation.EnableWebMvc;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.ResourceHandlerRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurerAdapter;
import springfox.documentation.swagger2.mappers.DirectModelMapperImpl;
import springfox.documentation.swagger2.mappers.ModelMapper;

import ru.yandex.direct.common.configuration.LoggingConfiguration;
import ru.yandex.direct.common.logging.LoggingConfigurerInterceptor;
import ru.yandex.direct.common.logging.LoggingSettings;
import ru.yandex.direct.common.metrics.MetricsFilter;
import ru.yandex.direct.common.metrics.MetricsInterceptor;
import ru.yandex.direct.common.tracing.TraceContextFilter;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.utils.json.LocalDateDeserializer;
import ru.yandex.direct.utils.json.LocalDateSerializer;
import ru.yandex.direct.web.core.WebLocaleResolver;
import ru.yandex.direct.web.core.configuration.SecurityConfiguration;
import ru.yandex.direct.web.core.configuration.SwaggerConfiguration;
import ru.yandex.direct.web.core.exception.DirectWebExceptionFilter;
import ru.yandex.direct.web.core.security.WebLocaleResolverFilter;
import ru.yandex.direct.web.core.security.authentication.DirectWebAuthenticationInterceptor;
import ru.yandex.direct.web.core.security.authentication.WebAuthenticationFilter;
import ru.yandex.direct.web.core.security.captcha.CaptchaFirewallInterceptor;
import ru.yandex.direct.web.core.security.csrf.CsrfInterceptor;
import ru.yandex.direct.web.core.security.netacl.NetAclInterceptor;
import ru.yandex.direct.web.core.semaphore.RedisSemaphoreInterceptor;
import ru.yandex.direct.web.logging.WebAuthenticationInfoLoggingFilter;
import ru.yandex.direct.web.logging.WebLoggingFilter;

import static ru.yandex.direct.web.core.security.configuration.BlackboxWebAuthenticationConfiguration.PARTNER_BASE_URL;
import static ru.yandex.direct.web.core.security.configuration.BlackboxWebAuthenticationConfiguration.PUBLIC_BASE_URL;

@EnableWebMvc
@ComponentScan(
        basePackages = {
                "ru.yandex.direct.web.core.exception",
                "ru.yandex.direct.web.core.semaphore",
                "ru.yandex.direct.web.logging"
        },
        excludeFilters = @ComponentScan.Filter(value = Configuration.class, type = FilterType.ANNOTATION)
)
@Import({
        DirectWebConfiguration.class,
        SecurityConfiguration.class,
        SwaggerConfiguration.class,
        UserActionLogConfiguration.class,
        AspectsConfiguration.class,
        LoggingConfiguration.class,
})
public class DirectWebAppConfiguration extends WebMvcConfigurerAdapter {

    @Autowired
    private CsrfInterceptor csrfInterceptor;

    @Autowired
    private CaptchaFirewallInterceptor captchaFirewallInterceptor;

    @Autowired
    private DirectWebAuthenticationInterceptor directWebAuthenticationInterceptor;

    @Autowired
    private RedisSemaphoreInterceptor requestsLimitSemaphore;

    @Autowired
    private MetricsInterceptor metricsInterceptor;

    @Autowired
    private NetAclInterceptor netAclInterceptor;

    @Value("${springfox.documentation.swagger.ui.baseurl}")
    private String swaggerUiBaseUrl;


    @Bean
    @Primary
    public ModelMapper modelMapper() {
        return new DirectModelMapperImpl();
    }

    @Bean(name = "directWebFilter")
    public Filter directWebFilter(TraceContextFilter traceContextFilter,
                                  WebAuthenticationFilter webAuthenticationFilter,
                                  DirectWebExceptionFilter directWebExceptionFilter,
                                  MetricsFilter metricsFilter,
                                  WebLoggingFilter webLoggingFilter,
                                  WebAuthenticationInfoLoggingFilter webAuthenticationInfoLoggingFilter) {
        CompositeFilter compositeFilter = new CompositeFilter();

        compositeFilter.setFilters(Arrays.asList(
                webLocaleResolverFilter(),
                characterEncodingFilter(),

                traceContextFilter,
                webLoggingFilter,
                metricsFilter,

                directWebExceptionFilter,
                webAuthenticationFilter,
                webAuthenticationInfoLoggingFilter
        ));
        return compositeFilter;
    }

    @Bean
    public LoggingSettings loggingDefaults() {
        return new LoggingSettings(1024 * 1024, 1024 * 1024);
    }

    @Bean
    public LoggingConfigurerInterceptor loggingConfigurerInterceptor() {
        return new LoggingConfigurerInterceptor(loggingDefaults());
    }

    @Bean
    public WebLocaleResolverFilter webLocaleResolverFilter() {
        return new WebLocaleResolverFilter();
    }

    @Bean
    public CharacterEncodingFilter characterEncodingFilter() {
        CharacterEncodingFilter encodingFilter = new CharacterEncodingFilter("UTF-8");
        encodingFilter.setForceRequestEncoding(true);
        encodingFilter.setForceResponseEncoding(true);
        return encodingFilter;
    }

    @Bean
    public LocaleResolver localeResolver() {
        return new WebLocaleResolver();
    }


    @Override
    public void addInterceptors(InterceptorRegistry registry) {
        final String pathPattern = "/**";

        final String publicPattern = PUBLIC_BASE_URL + "**";
        final String partnerPattern = PARTNER_BASE_URL + "**";
        final String swaggerUiPattern = swaggerUiBaseUrl + "/**";
        final String swaggerResourcesPattern = "/swagger-resources/**";
        final String[] excludePathPatterns = new String[]{swaggerUiPattern, swaggerResourcesPattern};

        registry.addInterceptor(loggingConfigurerInterceptor())
                .addPathPatterns(pathPattern)
                .excludePathPatterns(excludePathPatterns);

        registry.addInterceptor(metricsInterceptor)
                .addPathPatterns(pathPattern)
                .excludePathPatterns(excludePathPatterns);

        registry.addInterceptor(requestsLimitSemaphore)
                .addPathPatterns(pathPattern)
                .excludePathPatterns(excludePathPatterns);

        registry.addInterceptor(csrfInterceptor)
                .addPathPatterns(pathPattern)
                .excludePathPatterns(excludePathPatterns)
                .excludePathPatterns(publicPattern, partnerPattern);

        registry.addInterceptor(captchaFirewallInterceptor)
                .addPathPatterns(pathPattern)
                .excludePathPatterns(excludePathPatterns);

        registry.addInterceptor(directWebAuthenticationInterceptor)
                .addPathPatterns(pathPattern)
                .excludePathPatterns(excludePathPatterns)
                .excludePathPatterns(publicPattern, partnerPattern);

        registry.addInterceptor(netAclInterceptor)
                .addPathPatterns(pathPattern);
    }

    @Override
    public void extendMessageConverters(List<HttpMessageConverter<?>> converters) {
        // Запрещаем преобразовывать float-литералы в int, long, BigInteger поля для mediatype: application/json
        MappingJackson2HttpMessageConverter mappingJackson2HttpMessageConverter = StreamEx.of(converters)
                .findFirst(MappingJackson2HttpMessageConverter.class::isInstance)
                .map(MappingJackson2HttpMessageConverter.class::cast)
                .orElse(null);

        if (mappingJackson2HttpMessageConverter != null) {
            Module dateTimeModule = JsonUtils.createLocalDateTimeModule();

            SimpleModule dateModule = new SimpleModule(DirectWebAppConfiguration.class.getName());
            dateModule.addSerializer(LocalDate.class, new LocalDateSerializer());
            dateModule.addDeserializer(LocalDate.class, new LocalDateDeserializer());

            mappingJackson2HttpMessageConverter.getObjectMapper()
                    .registerModule(dateTimeModule)
                    .registerModule(dateModule)
                    .disable(DeserializationFeature.ACCEPT_FLOAT_AS_INT);
        }
    }

    @Override
    public void addResourceHandlers(ResourceHandlerRegistry registry) {
        /*
        Расширяем static и webjars для внутренних инструментов,
        чтобы можно было использовать относительные пути например для js библиотек
         */
        registry
                .addResourceHandler("/webjars/**",
                        "/internal_tools/webjars/**",
                        "/grid/webjars/**",
                        PUBLIC_BASE_URL + "grid/webjars/**",
                        "/user-action-log/webjars/**")
                .addResourceLocations("classpath:/META-INF/resources/webjars/");
        registry
                .addResourceHandler("/static/**",
                        "/internal_tools/static/**",
                        "/grid/static/**",
                        PUBLIC_BASE_URL + "grid/static/**",
                        "/user-action-log/static/**")
                .addResourceLocations("/static/");
    }
}
