package ru.yandex.direct.web.entity.adgeneration.controller;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import NAdvMachine.Searchqueryrec;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import one.util.streamex.StreamEx;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import ru.yandex.direct.bangenproxy.client.model.TextInfoCombinatorics;
import ru.yandex.direct.bangenproxy.client.model.TextInfoUc;
import ru.yandex.direct.core.entity.adgeneration.ImageGenerationService;
import ru.yandex.direct.core.entity.adgeneration.KeywordGenerationService;
import ru.yandex.direct.core.entity.adgeneration.RegionGenerationService;
import ru.yandex.direct.core.entity.adgeneration.SitelinkGenerationService;
import ru.yandex.direct.core.entity.adgeneration.TimezoneGenerationService;
import ru.yandex.direct.core.entity.adgeneration.model.ImageSuggest;
import ru.yandex.direct.core.entity.adgeneration.model.RegionSuggest;
import ru.yandex.direct.core.entity.adgeneration.model.SitelinkSuggest;
import ru.yandex.direct.core.entity.adgeneration.region.InputContainer;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.security.authorization.PreAuthorizeRead;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.richcontent.model.UrlInfo;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectId;
import ru.yandex.direct.validation.result.DefectInfo;
import ru.yandex.direct.web.annotations.CheckSubjectAndUserFeaturesSeparately;
import ru.yandex.direct.web.annotations.OperatorHasFeatures;
import ru.yandex.direct.web.annotations.SubjectHasFeatures;
import ru.yandex.direct.web.core.model.WebErrorResponse;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.model.WebSuccessResponse;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;
import ru.yandex.direct.web.entity.adgeneration.model.GenerateKeywordsBySnippetParams;
import ru.yandex.direct.web.entity.adgeneration.model.apply.ApplyGenerationResults;
import ru.yandex.direct.web.entity.adgeneration.model.response.WebGenerateCustomTextResult;
import ru.yandex.direct.web.entity.adgeneration.model.response.WebGenerateImageResult;
import ru.yandex.direct.web.entity.adgeneration.model.response.WebGenerateKeywordsResult;
import ru.yandex.direct.web.entity.adgeneration.model.response.WebGenerateRegionResult;
import ru.yandex.direct.web.entity.adgeneration.model.response.WebGenerateSitelinkResult;
import ru.yandex.direct.web.entity.adgeneration.model.response.WebGenerateTextResult;
import ru.yandex.direct.web.entity.adgeneration.model.response.WebGenerateTextSuggestionsResult;
import ru.yandex.direct.web.entity.adgeneration.model.response.WebGenerateTimezoneResult;

import static ru.yandex.direct.core.entity.adgeneration.model.GenerationDefectIds.CAMPAIGN_WITHOUT_HREF;
import static ru.yandex.direct.feature.FeatureName.NEW_TEXT_SUGGESTIONS_GENERATION_ENABLED;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.web.core.security.authentication.DirectCookieAuthProvider.PARAMETER_ULOGIN;

@Controller
@ParametersAreNonnullByDefault
@RequestMapping(value = "ad_generation",
        produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Api(tags = "adGeneration")
public class AdGenerationController {

    private static final String CAMPAIGN_KEY = "campaignId";
    private static final String AD_GROUP_KEY = "adGroupId";
    private static final String WARNING_KEY = "warnings";
    private static final String URL_SOURCE_KEY = "UrlSource";
    private static final String SOURCE_BANNER = "Banner";
    private static final String SOURCE_CAMPAIGN = "Campaign";
    private static final String ALG_VERSION = "algVersion";

    private static final Logger logger = LoggerFactory.getLogger(AdGenerationController.class);

    private final KeywordGenerationService keywordGenerationService;
    private final RegionGenerationService regionGenerationService;
    private final ImageGenerationService imageGenerationService;
    private final SitelinkGenerationService sitelinkGenerationService;
    private final TimezoneGenerationService timezoneGenerationService;
    private final DirectWebAuthenticationSource authenticationSource;
    private final ShardHelper shardHelper;
    private final CampaignRepository campaignRepository;
    private final FeatureService featureService;

    @Autowired
    public AdGenerationController(KeywordGenerationService keywordGenerationService,
                                  RegionGenerationService regionGenerationService,
                                  ImageGenerationService imageGenerationService,
                                  SitelinkGenerationService sitelinkGenerationService,
                                  TimezoneGenerationService timezoneGenerationService,
                                  DirectWebAuthenticationSource authenticationSource,
                                  ShardHelper shardHelper,
                                  CampaignRepository campaignRepository,
                                  FeatureService featureService) {
        this.keywordGenerationService = keywordGenerationService;
        this.regionGenerationService = regionGenerationService;
        this.imageGenerationService = imageGenerationService;
        this.sitelinkGenerationService = sitelinkGenerationService;
        this.timezoneGenerationService = timezoneGenerationService;
        this.authenticationSource = authenticationSource;
        this.shardHelper = shardHelper;
        this.campaignRepository = campaignRepository;
        this.featureService = featureService;
    }

    @ApiOperation(
            value = "Generate keywords by snippet and user keywords",
            httpMethod = "POST",
            nickname = "generateKeywordsBySnippet"
    )
    @ApiResponses({
            @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
            @ApiResponse(code = 200, message = "Ok", response = WebGenerateKeywordsResult.class)
    })
    @RequestMapping(path = "ad_generation/generate_keywords_by_snippet", method = RequestMethod.POST)
    @ResponseBody
    @PreAuthorizeRead
    @CheckSubjectAndUserFeaturesSeparately
    @OperatorHasFeatures({FeatureName.SUGGEST_GENERATED_GROUP_PHRASES_FOR_OPERATOR})
    @OperatorHasFeatures({FeatureName.SUGGEST_GENERATED_PHRASES_BY_SNIPPET_FOR_OPERATOR})
    @SubjectHasFeatures({FeatureName.UC_SUGGEST_AUDIENCE_STEP_PHRASES})
    public WebResponse generateKeywordsBySnippet(
            @RequestBody GenerateKeywordsBySnippetParams generateKeywordsBySnippetParams,
            @RequestParam(required = false) Long campaignId,
            @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin
    ) {
        Map<String, Object> additionalInfo = new HashMap<>();
        additionalInfo.put(CAMPAIGN_KEY, campaignId);

        Searchqueryrec.TSearchQueryRecRequest.Builder builder = Searchqueryrec.TSearchQueryRecRequest.newBuilder();
        if (StringUtils.isNotBlank(generateKeywordsBySnippetParams.getTitle())) {
            builder.setBannerTitle(generateKeywordsBySnippetParams.getTitle());
        }

        if (StringUtils.isNotBlank(generateKeywordsBySnippetParams.getBody())) {
            builder.setBannerText(generateKeywordsBySnippetParams.getBody());
        }

        if (StringUtils.isNotBlank(generateKeywordsBySnippetParams.getUrl())) {
            builder.setBannerURL(generateKeywordsBySnippetParams.getUrl());
            additionalInfo.put(URL_SOURCE_KEY, SOURCE_BANNER);
        } else if (campaignId != null) {
            int shard = shardHelper.getShardByCampaignId(campaignId);
            String urlFromDb = campaignRepository.getHrefCampAdditionalData(shard, List.of(campaignId)).get(campaignId);
            if (StringUtils.isNotBlank(urlFromDb)) {
                builder.setBannerURL(urlFromDb);
                additionalInfo.put(URL_SOURCE_KEY, SOURCE_CAMPAIGN);
            }
        } else if (StringUtils.isBlank(generateKeywordsBySnippetParams.getTitle()) &&
                StringUtils.isBlank(generateKeywordsBySnippetParams.getBody())) {
            logger.error("empty request");

            return new WebGenerateKeywordsResult(null, Map.of(WARNING_KEY, List.of(CAMPAIGN_WITHOUT_HREF.getCode(), "empty request")));
        }

        if (CollectionUtils.isNotEmpty(generateKeywordsBySnippetParams.getPlusWords()) ||
                CollectionUtils.isNotEmpty(generateKeywordsBySnippetParams.getMinusWords())) {
            Searchqueryrec.TSearchQueryRecRequest.TPhrases.Builder phrasesBuilder =
                    Searchqueryrec.TSearchQueryRecRequest.TPhrases.newBuilder();

            if (CollectionUtils.isNotEmpty(generateKeywordsBySnippetParams.getPlusWords())) {
                phrasesBuilder.addAllPlus(generateKeywordsBySnippetParams.getPlusWords());
            }

            if (CollectionUtils.isNotEmpty(generateKeywordsBySnippetParams.getMinusWords())) {
                phrasesBuilder.addAllMinus(generateKeywordsBySnippetParams.getMinusWords());
            }
            builder.setPhrases(phrasesBuilder.build());
        }

        if (CollectionUtils.isNotEmpty(generateKeywordsBySnippetParams.getGeoIds())) {
            var plusRegionIds = filterList(generateKeywordsBySnippetParams.getGeoIds(),
                    rid -> rid >= 0);
            var minusRegionIds = filterAndMapList(generateKeywordsBySnippetParams.getGeoIds(),
                    rid -> rid < 0, Math::abs);
            builder.addAllRegionIds(plusRegionIds);
            builder.addAllMinusRegionIds(minusRegionIds);
        }

        Result<Collection<Keyword>> generationResult =
                keywordGenerationService.generateKeywords(builder.build(), additionalInfo);
        var keywords = mapList(generationResult.getResult(), Keyword::getPhrase);

        logger.info("Suggested keywords: {}", String.join(", ", keywords));

        additionalInfo.put(WARNING_KEY, getWarningList(generationResult));
        return new WebGenerateKeywordsResult(keywords, additionalInfo);
    }

    @ApiOperation(
            value = "Generate ad text by url",
            httpMethod = "GET",
            nickname = "generateText"
    )
    @ApiResponses({
            @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
            @ApiResponse(code = 200, message = "Ok", response = WebGenerateTextResult.class)
    })
    @RequestMapping(path = "ad_generation/generate_text", method = RequestMethod.GET)
    @ResponseBody
    @PreAuthorizeRead
    @CheckSubjectAndUserFeaturesSeparately
    @OperatorHasFeatures({FeatureName.SUGGEST_GENERATED_TITLE_AND_SNIPPET_FOR_OPERATOR})
    @SubjectHasFeatures({FeatureName.UC_SUGGEST_ADS_STEP_SNIPPET})
    public WebResponse generateText(
            @RequestParam Long campaignId,
            @RequestParam(required = false) Long adGroupId,
            @RequestParam(required = false) Long groupId,
            @RequestParam(required = false) String url,
            @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin
    ) {
        ClientId clientId = authenticationSource.getAuthentication().getSubjectUser().getClientId();
        boolean userHasNewGenerationFeature = featureService.getEnabledForClientId(clientId)
                .contains(NEW_TEXT_SUGGESTIONS_GENERATION_ENABLED.getName());
        Map<String, Object> additionalInfo = new HashMap<>();
        Result<?> urlInfoResult;
        String title = null;
        String body = null;
        if (userHasNewGenerationFeature) {
            urlInfoResult = keywordGenerationService.getSingleTextSuggestionsByUrl(url, campaignId, additionalInfo);
            TextInfoUc textInfo = (TextInfoUc) urlInfoResult.getResult();
            if (textInfo != null) {
                title = textInfo.getTitle();
                body = textInfo.getBody();
                additionalInfo.put(ALG_VERSION, textInfo.getAlgVersion());
            }
        } else {
            urlInfoResult = keywordGenerationService.getUrlInfo(url, campaignId, additionalInfo);
            UrlInfo urlInfo = (UrlInfo) urlInfoResult.getResult();
            title = urlInfo == null ? null : urlInfo.getTitle();
            body = urlInfo == null ? null : urlInfo.getDescription();
        }
        additionalInfo.put(WARNING_KEY, getWarningList(urlInfoResult));
        additionalInfo.put(CAMPAIGN_KEY, campaignId);
        if (groupId != null && groupId != 0) {
            additionalInfo.put(AD_GROUP_KEY, groupId);
        } else {
            additionalInfo.put(AD_GROUP_KEY, adGroupId);
        }

        return new WebGenerateTextResult(title, body, additionalInfo);
    }

    @ApiOperation(
            value = "Generate several ad texts suggestions by url",
            httpMethod = "GET",
            nickname = "generateTextSuggestions"
    )
    @RequestMapping(path = "ad_generation/generate_text_suggestions", method = RequestMethod.GET)
    @ResponseBody
    @PreAuthorizeRead
    @ApiResponses({
            @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
            @ApiResponse(code = 200, message = "Ok", response = WebGenerateTextSuggestionsResult.class)
    })
    public WebResponse generateTextSuggestions(
            @RequestParam(required = false) Long campaignId,
            @RequestParam(required = false) String url,
            @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin
    ) {
        Map<String, Object> additionalInfo = new HashMap<>();
        Result<TextInfoCombinatorics> textForCombinatorics = keywordGenerationService
                .getMultipleTextSuggestionsByUrl(url, campaignId, null, additionalInfo);
        additionalInfo.put(WARNING_KEY, getWarningList(textForCombinatorics));
        TextInfoCombinatorics textInfo = textForCombinatorics.getResult();
        String previewTitle = null;
        List<String> titles = null;
        List<String> bodies = null;
        if (textInfo != null) {
            titles = textInfo.getTitle();
            bodies = textInfo.getBody();
            additionalInfo.put(ALG_VERSION, textInfo.getAlgVersion());
            previewTitle = textInfo.getRcaTitle();
        }

        if (titles != null) {
            logger.info("Generated {} titles for campaign {}: '{}'",
                    titles.size(), nvl(campaignId, 0L), String.join("', '", titles));
        }
        if (bodies != null) {
            logger.info("Generated {} bodies for campaign {}: '{}'",
                    bodies.size(), nvl(campaignId, 0L), String.join("', '", bodies));
        }
        return new WebGenerateTextSuggestionsResult(previewTitle, titles, bodies, additionalInfo);
    }

    @ApiOperation(
            value = "Generate ad texts suggestions based on url and user's text",
            httpMethod = "GET",
            nickname = "generateTextSuggestionsBySubject"
    )
    @RequestMapping(path = "ad_generation/generate_text_suggestions_by_subject", method = RequestMethod.GET)
    @ResponseBody
    @PreAuthorizeRead
    @ApiResponses({
            @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
            @ApiResponse(code = 200, message = "Ok", response = WebGenerateCustomTextResult.class)
    })
    public WebResponse generateTextSuggestionsBySubject(
            @RequestParam(required = false) Long campaignId,
            @RequestParam(required = false) String url,
            @RequestParam(required = false) String adSubject,
            @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin
    ) {
        Map<String, Object> additionalInfo = new HashMap<>();
        Result<TextInfoCombinatorics> textInfoResult = keywordGenerationService
                .getMultipleTextSuggestionsByUrl(url, campaignId, adSubject, additionalInfo);

        additionalInfo.put(WARNING_KEY, getWarningList(textInfoResult));
        TextInfoCombinatorics textInfo = textInfoResult.getResult();
        List<String> titles = null;
        List<String> bodies = null;
        if (textInfo != null) {
            titles = textInfo.getTitle();
            bodies = textInfo.getBody();
            additionalInfo.put(ALG_VERSION, textInfo.getAlgVersion());
        }

        return new WebGenerateCustomTextResult(titles, bodies, adSubject, additionalInfo);
    }

    @ApiOperation(
            value = "Generate adisNotBlank regions",
            httpMethod = "GET",
            nickname = "generateRegions"
    )
    @ApiResponses({
            @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
            @ApiResponse(code = 200, message = "Ok", response = WebGenerateRegionResult.class)
    })
    @RequestMapping(path = "ad_generation/generate_regions", method = RequestMethod.GET)
    @ResponseBody
    @PreAuthorizeRead
    @CheckSubjectAndUserFeaturesSeparately
    @OperatorHasFeatures({FeatureName.SUGGEST_GENERATED_REGIONS_FOR_OPERATOR})
    @SubjectHasFeatures({FeatureName.UC_SUGGEST_AUDIENCE_STEP_GEO})
    public WebResponse generateRegions(
            @RequestParam(required = false) Long campaignId,
            @RequestParam(required = false) Long permalinkId,
            @RequestParam(required = false) Long counterId,
            @RequestParam(required = false) String url,
            @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin
    ) {
        InputContainer input = new InputContainer();
        if (campaignId != null) {
            input.put(InputContainer.CAMPAIGN_ID, campaignId);
        }
        if (permalinkId != null) {
            input.put(InputContainer.PERMALINK_ID, permalinkId);
        }
        if (counterId != null) {
            input.put(InputContainer.COUNTER_IDS, Set.of(counterId));
        }
        if (StringUtils.isNotEmpty(url)) {
            input.put(InputContainer.URL, url);
        }
        ClientId clientId = authenticationSource.getAuthentication().getSubjectUser().getClientId();
        Map<String, Object> additionalInfo = new HashMap<>();
        Result<Collection<RegionSuggest>> suggests =
                regionGenerationService.generateRegions(clientId, input, additionalInfo, null);
        StreamEx.of(
                InputContainer.CAMPAIGN_ID,
                InputContainer.PERMALINK_ID,
                InputContainer.COUNTER_IDS,
                InputContainer.URL
        ).forEach(key -> additionalInfo.put(key, input.get(key)));
        additionalInfo.put(WARNING_KEY, getWarningList(suggests));

        return new WebGenerateRegionResult(suggests.getResult(), additionalInfo);
    }

    @ApiOperation(
            value = "Generate ad images",
            httpMethod = "GET",
            nickname = "generateImages"
    )
    @ApiResponses({
            @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
            @ApiResponse(code = 200, message = "Ok", response = WebGenerateImageResult.class)
    })
    @GetMapping(path = "ad_generation/generate_images")
    @ResponseBody
    @PreAuthorizeRead
    @CheckSubjectAndUserFeaturesSeparately
    @OperatorHasFeatures({FeatureName.SUGGEST_GENERATED_IMAGES_FOR_OPERATOR})
    @SubjectHasFeatures({FeatureName.UC_SUGGEST_ADS_STEP_IMAGES})
    public WebResponse generateImages(
            @RequestParam Long campaignId,
            @RequestParam(required = false) Long adGroupId,
            @RequestParam(required = false) Long groupId,
            @RequestParam(required = false) String url,
            @RequestParam(required = false) String text,
            @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin
    ) {

        ClientId clientId = authenticationSource.getAuthentication().getSubjectUser().getClientId();
        Map<String, Object> additionalInfo = new HashMap<>();
        Result<Collection<ImageSuggest>> suggests;

        if (groupId != null && groupId != 0) {
            suggests = imageGenerationService.generateImages(clientId, url, text,
                    campaignId, groupId, additionalInfo);
            additionalInfo.put(AD_GROUP_KEY, groupId);
        } else {
            suggests = imageGenerationService.generateImages(clientId, url, text,
                    campaignId, adGroupId, additionalInfo);
            additionalInfo.put(AD_GROUP_KEY, adGroupId);
        }

        additionalInfo.put(CAMPAIGN_KEY, campaignId);
        additionalInfo.put(WARNING_KEY, getWarningList(suggests));
        return new WebGenerateImageResult(
                suggests.getResult(),
                additionalInfo);
    }

    @ApiOperation(
            value = "Generate ad sitelinks",
            httpMethod = "GET",
            nickname = "generateSitelinks"
    )
    @ApiResponses({
            @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
            @ApiResponse(code = 200, message = "Ok", response = WebGenerateSitelinkResult.class)
    })
    @RequestMapping(path = "ad_generation/generate_sitelinks", method = RequestMethod.GET)
    @ResponseBody
    @PreAuthorizeRead
    @CheckSubjectAndUserFeaturesSeparately
    @OperatorHasFeatures({FeatureName.SUGGEST_GENERATED_SITELINKS_FOR_OPERATOR})
    @SubjectHasFeatures({FeatureName.UC_SUGGEST_ADS_STEP_SITELINKS})
    public WebResponse generateSitelinks(
            @RequestParam(required = false) String url,
            @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin
    ) {

        Map<String, Object> additionalInfo = new HashMap<>();
        Result<Collection<SitelinkSuggest>> suggests = sitelinkGenerationService.generateSitelinks(url, additionalInfo);
        additionalInfo.put(WARNING_KEY, getWarningList(suggests));

        return new WebGenerateSitelinkResult(suggests.getResult(), additionalInfo);
    }

    @ApiOperation(
            value = "Save results of using of generation data",
            httpMethod = "POST",
            nickname = "saveUsingResults"
    )
    @ApiResponses({
            @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
            @ApiResponse(code = 200, message = "Ok", response = WebResponse.class)
    })
    @PostMapping(path = "ad_generation/save_results")
    @ResponseBody
    @PreAuthorizeRead
    public WebResponse saveGenerationDataUsingResults(
            @RequestParam(value = "UC", required = false, defaultValue = "false") Boolean isUniversalCampaign,
            @RequestBody ApplyGenerationResults applyGenerationResults,
            @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin
    ) {
        String message = isUniversalCampaign ? "Apply UC {} generation: {}" : "Apply {} generation: {}";
        if (applyGenerationResults.getApplyKeywordsGeneration() != null) {
            logger.info(message,
                    "keywords",
                    JsonUtils.toJson(applyGenerationResults.getApplyKeywordsGeneration()));
        }
        if (applyGenerationResults.getApplyImagesGeneration() != null) {
            logger.info(message,
                    "images",
                    JsonUtils.toJson(applyGenerationResults.getApplyImagesGeneration()));
        }
        if (applyGenerationResults.getApplyTextGeneration() != null) {
            logger.info(message,
                    "text",
                    JsonUtils.toJson(applyGenerationResults.getApplyTextGeneration().withDefaults()));
        }
        if (applyGenerationResults.getApplyRegionsGeneration() != null) {
            logger.info(message,
                    "regions",
                    JsonUtils.toJson(applyGenerationResults.getApplyRegionsGeneration()));
        }
        if (applyGenerationResults.getApplySitelinksGeneration() != null) {
            logger.info(message,
                    "sitelinks",
                    JsonUtils.toJson(applyGenerationResults.getApplySitelinksGeneration()));
        }
        return new WebSuccessResponse();
    }

    @ApiOperation(
            value = "Generate time target timezone",
            httpMethod = "GET",
            nickname = "generateTimezone"
    )
    @ApiResponses({
            @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
            @ApiResponse(code = 200, message = "Ok", response = WebGenerateRegionResult.class)
    })
    @RequestMapping(path = "ad_generation/generate_timezone", method = RequestMethod.GET)
    @ResponseBody
    @PreAuthorizeRead
    public WebResponse generateRegions(
            @RequestParam(required = false) Long regionId,
            @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin
    ) {
        Long timezoneId = timezoneGenerationService.generateTimezoneId(regionId);
        return new WebGenerateTimezoneResult(timezoneId);
    }

    private static List<String> getWarningList(Result<?> result) {
        return StreamEx.of(result.getWarnings())
                .map(DefectInfo::getDefect)
                .map(Defect::defectId)
                .map(DefectId::getCode)
                .toList();
    }

}
