package ru.yandex.direct.web.entity.adgroup.controller;

import java.util.List;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.direct.core.entity.adgroup.container.UntypedAdGroup;
import ru.yandex.direct.core.entity.banner.service.BannerService;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.result.DefaultPathNodeConverterProvider;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;
import ru.yandex.direct.web.core.model.WebErrorResponse;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.model.WebSuccessResponse;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;
import ru.yandex.direct.web.entity.adgroup.model.SaveAdGroupsResponse;
import ru.yandex.direct.web.entity.adgroup.model.WebTextAdGroup;
import ru.yandex.direct.web.entity.adgroup.presentations.AdGroupPathConverters;
import ru.yandex.direct.web.entity.adgroup.service.text.AddTextAdGroupService;
import ru.yandex.direct.web.entity.adgroup.service.text.UpdateTextAdGroupService;
import ru.yandex.direct.web.validation.kernel.ValidationResultConversionService;

import static ru.yandex.direct.core.entity.domain.DomainUtils.refineDomain;
import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.web.core.security.authentication.DirectCookieAuthProvider.PARAMETER_ULOGIN;
import static ru.yandex.direct.web.entity.adgroup.presentations.AdGroupPathConverters.TEXT_AD_GROUP_PATH_CONVERTER;

@RestController
@RequestMapping(path = "adgroups",
        consumes = MediaType.APPLICATION_JSON_UTF8_VALUE,
        produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Api(tags = "adGroups")
public class AdGroupController {

    private final AddTextAdGroupService addTextAdGroupService;
    private final UpdateTextAdGroupService updateTextAdGroupService;
    private final BannerService bannerService;
    private final ValidationResultConversionService validationResultConversionService;
    private final PathNodeConverterProvider pathNodeConverterProvider;
    private final DirectWebAuthenticationSource authenticationSource;
    private final AdGroupControllerHelper adGroupControllerHelper;

    public AdGroupController(AddTextAdGroupService addTextAdGroupService,
                             UpdateTextAdGroupService updateTextAdGroupService,
                             BannerService bannerService,
                             ValidationResultConversionService validationResultConversionService,
                             PathNodeConverterProvider pathNodeConverterProvider,
                             DirectWebAuthenticationSource authenticationSource,
                             AdGroupControllerHelper adGroupControllerHelper) {
        this.addTextAdGroupService = addTextAdGroupService;
        this.updateTextAdGroupService = updateTextAdGroupService;
        this.bannerService = bannerService;
        this.validationResultConversionService = validationResultConversionService;
        this.authenticationSource = authenticationSource;

        this.pathNodeConverterProvider = DefaultPathNodeConverterProvider.builder()
                .register(UntypedAdGroup.class, TEXT_AD_GROUP_PATH_CONVERTER)
                .register(TargetInterest.class, AdGroupPathConverters.AD_GROUP_RETARGETING_PATH_CONVERTER)
                .register(RelevanceMatch.class, AdGroupPathConverters.AD_GROUP_RELEVANCE_MATCH_PATH_CONVERTER)
                .fallbackTo(pathNodeConverterProvider)
                .build();
        this.adGroupControllerHelper = adGroupControllerHelper;
    }

    @ApiOperation(value = "save", httpMethod = "POST", nickname = "save")
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Forbidden", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = WebSuccessResponse.class)
            }
    )
    @PreAuthorizeWrite
    @RequestMapping(path = "/text/save", method = RequestMethod.POST)
    public WebResponse saveTextAdGroup(
            @RequestBody List<WebTextAdGroup> adGroups,
            @RequestParam(value = "campaign_id") Long campaignId,
            @RequestParam(value = "is_new_adgroups") Boolean isNewAdGroups,
            @RequestParam(value = "save_draft") Boolean saveDraft,
            @RequestParam(value = "is_copy", required = false) Boolean isCopy,
            @RequestParam(value = "camp_banners_domain", required = false) String campaignBannerDomain,
            @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin) {
        User operator = authenticationSource.getAuthentication().getOperator();
        User subjectUser = authenticationSource.getAuthentication().getSubjectUser();
        ClientId clientId = subjectUser.getClientId();
        long clientUid = subjectUser.getUid();
        long operatorUid = operator.getUid();

        adGroupControllerHelper.checkRequestParams(operatorUid, clientId, campaignId, CampaignAccessType.READ_WRITE);

        // todo придумать, как лучше делать маппинг имен полей для полиморфных объектов, как группы и баннеры

        if (campaignBannerDomain != null) {
            // Валидировать домены от супера нужно до обновления их в add/update чтоб отфильтровать некорректный запрос
            campaignBannerDomain = refineDomain(campaignBannerDomain);
            AdGroupControllerUtils.validateDomainWithException(campaignBannerDomain);
        }

        MassResult<Long> result;
        if (isNewAdGroups) {
            isCopy = isCopy != null && isCopy;
            result = addTextAdGroupService
                    .addAdGroups(adGroups, campaignId, isCopy, saveDraft, operatorUid, clientId, clientUid);
        } else {
            result = updateTextAdGroupService
                    .updateAdGroups(adGroups, campaignId, operatorUid, clientId, clientUid, saveDraft);
        }

        if (campaignBannerDomain != null) {
            // Устанавливать домены от супера нужно после обновления их в add/update, иначе они будут взяты из href
            bannerService.changeCampaignBannersDomains(operatorUid, campaignId, campaignBannerDomain);
        }

        if (hasValidationIssues(result)) {
            return validationResultConversionService.buildValidationResponse(result.getValidationResult(),
                    pathNodeConverterProvider);
        } else {
            return new SaveAdGroupsResponse()
                    .withResult(mapList(result.getResult(), Result::getResult));
        }
    }
}
