package ru.yandex.direct.web.entity.adgroup.service.contentpromotionvideo;

import java.math.BigDecimal;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.ComplexContentPromotionAdGroup;
import ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupAddOperationFactory;
import ru.yandex.direct.core.entity.adgroup.service.complex.contentpromotion.ComplexContentPromotionAdGroupAddOperation;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionAutoPriceParams;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionFixedAutoPrices;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.entity.adgroup.converter.ContentPromotionAdGroupConverter;
import ru.yandex.direct.web.entity.adgroup.model.WebContentPromotionAdGroup;
import ru.yandex.direct.web.entity.adgroup.service.CopyAdGroupDataFiller;
import ru.yandex.direct.ytcore.entity.statistics.service.RecentStatisticsService;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Collections.emptyList;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Сервис для добавления групп объявлений продвижения контента.
 */
@Service
@ParametersAreNonnullByDefault
public class AddContentPromotionVideoAdGroupService {
    private final ComplexAdGroupAddOperationFactory complexAdGroupAddOperationFactory;
    private final ClientGeoService clientGeoService;
    private final ClientService clientService;
    private final WebContentPromotionVideoAdGroupValidationService webContentPromotionVideoAdGroupValidationService;
    private final RecentStatisticsService recentStatisticsService;
    private final CopyAdGroupDataFiller copyAdGroupDataFiller;
    private final ShardHelper shardHelper;

    public AddContentPromotionVideoAdGroupService(
            ComplexAdGroupAddOperationFactory complexAdGroupAddOperationFactory,
            ClientGeoService clientGeoService,
            WebContentPromotionVideoAdGroupValidationService webContentPromotionVideoAdGroupValidationService,
            ClientService clientService,
            RecentStatisticsService recentStatisticsService,
            CopyAdGroupDataFiller copyAdGroupDataFiller,
            ShardHelper shardHelper) {
        this.complexAdGroupAddOperationFactory = complexAdGroupAddOperationFactory;
        this.clientGeoService = clientGeoService;
        this.webContentPromotionVideoAdGroupValidationService = webContentPromotionVideoAdGroupValidationService;
        this.clientService = clientService;
        this.recentStatisticsService = recentStatisticsService;
        this.copyAdGroupDataFiller = copyAdGroupDataFiller;
        this.shardHelper = shardHelper;
    }

    public MassResult<Long> addAdGroups(List<WebContentPromotionAdGroup> adGroups, long campaignId,
                                        boolean isCopy, boolean saveDraft, long operatorUid, ClientId clientId,
                                        long clientUid) {
        Currency clientCurrency = clientService.getWorkCurrency(clientId);
        ValidationResult<List<WebContentPromotionAdGroup>, Defect> vr =
                webContentPromotionVideoAdGroupValidationService.validate(adGroups, clientCurrency);

        if (vr.hasAnyErrors()) {
            return MassResult.brokenMassAction(emptyList(), vr);
        }

        ShowConditionAutoPriceParams autoPriceParams;
        List<ComplexContentPromotionAdGroup> complexAdGroups = ContentPromotionAdGroupConverter
                .webAdGroupsToCoreComplexContentPromotionAdGroups(adGroups, campaignId);

        if (isCopy) {
            autoPriceParams = getAutoPriceParamsForCopy();
            fillDataOnCopy(clientId, complexAdGroups, adGroups);
        } else {
            autoPriceParams = getAutoPriceParamsForAdd(adGroups);
        }

        GeoTree geoTree = clientGeoService.getClientTranslocalGeoTree(clientId);

        ComplexContentPromotionAdGroupAddOperation addOperation = complexAdGroupAddOperationFactory
                .createContentPromotionAdGroupAddOperation(saveDraft, complexAdGroups,
                        geoTree, true, autoPriceParams, operatorUid, clientId, clientUid);

        return addOperation.prepareAndApply();
    }

    private void fillDataOnCopy(ClientId clientId,
                                List<ComplexContentPromotionAdGroup> complexContentPromotionAdGroups,
                                List<WebContentPromotionAdGroup> webAdGroups) {
        checkArgument(complexContentPromotionAdGroups.size() == webAdGroups.size());
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        List<Double> generalPrices = mapList(webAdGroups, WebContentPromotionAdGroup::getGeneralPrice);

        List<List<Keyword>> adGroupsKeywords =
                mapList(complexContentPromotionAdGroups, ComplexContentPromotionAdGroup::getKeywords);
        copyAdGroupDataFiller.fillKeywordPrices(shard, clientId, adGroupsKeywords, generalPrices);

        List<List<RelevanceMatch>> adGroupsRelevanceMatches =
                mapList(complexContentPromotionAdGroups, ComplexContentPromotionAdGroup::getRelevanceMatches);
        copyAdGroupDataFiller
                .fillRelevanceMatchPrices(shard, clientId, adGroupsRelevanceMatches, generalPrices);
    }

    /**
     * Получение параметров для автоматического выставления недостающих ставок.
     * <p>
     * Если в добавляемой группе есть параметр {@code generalPrice},
     * то указываем, что эту ставку нужно выставить всем условиям показов.
     * <p>
     * Тут ожидается, что добавлять можно только одну группу за запрос, т.к.
     * иначе механизм выставления фиксированных автоставок нужно переделать.
     * Поэтому этот механизм не используется при копировании групп,
     * так как за раз можно копировать несколько групп.
     */
    private ShowConditionAutoPriceParams getAutoPriceParamsForAdd(List<WebContentPromotionAdGroup> adGroups) {
        ShowConditionFixedAutoPrices fixedAdGroupAutoPrices =
                ShowConditionFixedAutoPrices.ofGlobalFixedPrice(
                        ifNotNull(adGroups.get(0).getGeneralPrice(), BigDecimal::valueOf)
                );
        return new ShowConditionAutoPriceParams(
                fixedAdGroupAutoPrices,
                recentStatisticsService
        );
    }

    private ShowConditionAutoPriceParams getAutoPriceParamsForCopy() {
        ShowConditionFixedAutoPrices fixedAdGroupAutoPrices =
                ShowConditionFixedAutoPrices.ofGlobalFixedPrice(null);
        return new ShowConditionAutoPriceParams(
                fixedAdGroupAutoPrices,
                recentStatisticsService
        );
    }
}
