package ru.yandex.direct.web.entity.adgroup.service.contentpromotionvideo;

import java.math.BigDecimal;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.ComplexContentPromotionAdGroup;
import ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupUpdateOperationFactory;
import ru.yandex.direct.core.entity.adgroup.service.complex.contentpromotion.ComplexContentPromotionAdGroupUpdateOperation;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionAutoPriceParams;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionFixedAutoPrices;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.entity.adgroup.converter.ContentPromotionAdGroupConverter;
import ru.yandex.direct.web.entity.adgroup.model.WebContentPromotionAdGroup;
import ru.yandex.direct.ytcore.entity.statistics.service.RecentStatisticsService;

import static java.util.Collections.emptyList;

/**
 * Сервис для обновления групп объявлений продвижения контента.
 */
@Service
@ParametersAreNonnullByDefault
public class UpdateContentPromotionVideoAdGroupService {
    private final ComplexAdGroupUpdateOperationFactory complexAdGroupUpdateOperationFactory;
    private final ClientGeoService clientGeoService;
    private final ClientService clientService;
    private final RecentStatisticsService recentStatisticsService;
    private final WebContentPromotionVideoAdGroupValidationService webContentPromotionVideoAdGroupValidationService;

    @Autowired
    public UpdateContentPromotionVideoAdGroupService(
            ComplexAdGroupUpdateOperationFactory complexAdGroupUpdateOperationFactory,
            ClientGeoService clientGeoService,
            ClientService clientService,
            RecentStatisticsService recentStatisticsService,
            WebContentPromotionVideoAdGroupValidationService webContentPromotionVideoAdGroupValidationService) {
        this.complexAdGroupUpdateOperationFactory = complexAdGroupUpdateOperationFactory;
        this.clientGeoService = clientGeoService;
        this.clientService = clientService;
        this.recentStatisticsService = recentStatisticsService;
        this.webContentPromotionVideoAdGroupValidationService = webContentPromotionVideoAdGroupValidationService;
    }

    public MassResult<Long> updateAdGroups(List<WebContentPromotionAdGroup> adGroups, long campaignId,
                                           long operatorUid, ClientId clientId, long clientUid, boolean saveDraft) {
        List<ComplexContentPromotionAdGroup> complexAdGroups = ContentPromotionAdGroupConverter
                .webAdGroupsToCoreComplexContentPromotionAdGroups(adGroups, campaignId);

        Currency clientCurrency = clientService.getWorkCurrency(clientId);
        ValidationResult<List<WebContentPromotionAdGroup>, Defect> vr =
                webContentPromotionVideoAdGroupValidationService.validate(adGroups, clientCurrency);
        if (vr.hasAnyErrors()) {
            return MassResult.brokenMassAction(emptyList(), vr);
        }

        ShowConditionAutoPriceParams autoPriceParams = getAutoPriceParams(adGroups);
        GeoTree geoTree = clientGeoService.getClientTranslocalGeoTree(clientId);


        ComplexContentPromotionAdGroupUpdateOperation updateOperation = complexAdGroupUpdateOperationFactory
                .createContentPromotionAdGroupUpdateOperation(complexAdGroups, geoTree, true,
                        autoPriceParams, operatorUid, clientId, clientUid, saveDraft);
        return updateOperation.prepareAndApply();
    }

    /**
     * Получение параметров для автоматического выставления недостающих ставок.
     * <p>
     * Указываем, что в каких-то группах нужно выставить фиксированную ставку
     * для новых/изменившихся условий показов.
     */
    private ShowConditionAutoPriceParams getAutoPriceParams(List<WebContentPromotionAdGroup> adGroups) {
        ShowConditionFixedAutoPrices showConditionFixedAutoPrices =
                ShowConditionFixedAutoPrices.ofPerAdGroupFixedPrices(
                        StreamEx.of(adGroups)
                                .filter(adGroup -> adGroup.getGeneralPrice() != null)
                                .mapToEntry(WebContentPromotionAdGroup::getId,
                                        adGroup -> BigDecimal.valueOf(adGroup.getGeneralPrice()))
                                .toMap()
                );
        return new ShowConditionAutoPriceParams(
                showConditionFixedAutoPrices,
                recentStatisticsService
        );
    }
}
