package ru.yandex.direct.web.entity.adgroup.service.cpm;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.ComplexAdGroup;
import ru.yandex.direct.core.entity.adgroup.container.ComplexCpmAdGroup;
import ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupAddOperationFactory;
import ru.yandex.direct.core.entity.adgroup.service.complex.cpm.ComplexCpmAdGroupAddOperation;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.CpmYndxFrontpageAdGroupPriceRestrictions;
import ru.yandex.direct.core.entity.currency.service.CpmYndxFrontpageCurrencyService;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionAutoPriceParams;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionFixedAutoPrices;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.web.entity.adgroup.model.WebCpmAdGroup;
import ru.yandex.direct.web.entity.adgroup.service.CopyAdGroupDataFiller;
import ru.yandex.direct.ytcore.entity.statistics.service.RecentStatisticsService;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Collections.emptyList;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.web.entity.adgroup.converter.CpmAdGroupConverter.webAdGroupsToCoreComplexCpmAdGroups;
import static ru.yandex.direct.web.entity.adgroup.service.cpm.CpmAdGroupHelper.getGeneralPrice;

@Service
public class AddCpmAdGroupService {

    private final ComplexAdGroupAddOperationFactory complexAdGroupAddOperationFactory;
    private final ClientGeoService clientGeoService;
    private final ClientService clientService;
    private final WebCpmAdGroupValidationService webCpmAdGroupValidationService;
    private final RecentStatisticsService recentStatisticsService;
    private final CopyAdGroupDataFiller copyAdGroupDataFiller;
    private final ShardHelper shardHelper;
    private final CpmYndxFrontpageCurrencyService cpmYndxFrontpageCurrencyService;
    private final CpmAdGroupHelper cpmAdGroupHelper;

    public AddCpmAdGroupService(
            ComplexAdGroupAddOperationFactory complexAdGroupAddOperationFactory,
            ClientGeoService clientGeoService,
            ClientService clientService,
            WebCpmAdGroupValidationService webCpmAdGroupValidationService,
            RecentStatisticsService recentStatisticsService,
            CopyAdGroupDataFiller copyAdGroupDataFiller,
            ShardHelper shardHelper,
            CpmYndxFrontpageCurrencyService cpmYndxFrontpageCurrencyService,
            CpmAdGroupHelper cpmAdGroupHelper) {
        this.complexAdGroupAddOperationFactory = complexAdGroupAddOperationFactory;
        this.clientGeoService = clientGeoService;
        this.clientService = clientService;
        this.webCpmAdGroupValidationService = webCpmAdGroupValidationService;
        this.recentStatisticsService = recentStatisticsService;
        this.copyAdGroupDataFiller = copyAdGroupDataFiller;
        this.shardHelper = shardHelper;
        this.cpmYndxFrontpageCurrencyService = cpmYndxFrontpageCurrencyService;
        this.cpmAdGroupHelper = cpmAdGroupHelper;
    }

    public MassResult<Long> addAdGroups(List<WebCpmAdGroup> adGroups, long campaignId,
                                        boolean isCopy, boolean saveDraft, long operatorUid, ClientId clientId,
                                        long clientUid) {
        Currency clientCurrency = clientService.getWorkCurrency(clientId);

        int shard = shardHelper.getShardByClientId(clientId);
        Campaign campaign = cpmAdGroupHelper.getCampaign(shard, campaignId);
        boolean autoBudget = campaign.getStrategy().isAutoBudget();

        var cpmYndxFrontpageCurrencyRestrictions =
                cpmAdGroupHelper.getCpmYndxFrontpagePriceRestrictions(shard, adGroups, clientCurrency, campaign);
        var validationData = new WebCpmAdGroupValidationService.ValidationData(
                clientCurrency, autoBudget, campaign.getType(), cpmYndxFrontpageCurrencyRestrictions);
        var vr = webCpmAdGroupValidationService.validate(adGroups, validationData);
        if (vr.hasAnyErrors()) {
            return MassResult.brokenMassAction(emptyList(), vr);
        }

        List<ComplexCpmAdGroup> complexAdGroups =
                webAdGroupsToCoreComplexCpmAdGroups(adGroups, campaignId, campaign.getType());

        BigDecimal defaultPrice = campaign.getType() == CampaignType.CPM_YNDX_FRONTPAGE ?
                getCpmYndxFrontpageDefaultPrice(complexAdGroups, shard, clientCurrency)
                : getCpmBannerDefaultPrice(clientCurrency);
        ShowConditionAutoPriceParams autoPriceParams;
        if (isCopy) {
            //Для автобюджетных кампаний типа cpm_yndx_frontpage хотим проставить в базе иные ставки,
            // чем для копируемой группы. Связано с тем, что для данного типа кампаний минимальная ставка в группах
            //зависит от гео, а не только от типа группы
            List<Double> generalPricesForCopy = (campaign.getType() == CampaignType.CPM_YNDX_FRONTPAGE && autoBudget) ?
                    getCpmYndxFrontpageGeneralPricesForCopy(cpmYndxFrontpageCurrencyRestrictions) : null;
            autoPriceParams = getAutoPriceParamsForCopy();
            fillDataOnCopy(clientId, complexAdGroups, adGroups, generalPricesForCopy);
        } else {
            autoPriceParams = getAutoPriceParamsForAdd(adGroups, defaultPrice);
        }

        GeoTree geoTree = clientGeoService.getClientTranslocalGeoTree(clientId);

        ComplexCpmAdGroupAddOperation addOperation = complexAdGroupAddOperationFactory
                .createCpmAdGroupAddOperation(saveDraft, complexAdGroups, geoTree,
                        true, autoPriceParams, operatorUid, clientId, clientUid, true);
        return addOperation.prepareAndApply();
    }

    /**
     * Заполнение данных при копировании : price, priceContext, autobudgetPriority
     *
     * @param clientId                идентификатор клиента
     * @param complexCpmAdGroups      список ядровых групп
     * @param webAdGroups             список групп (модели web)
     * @param overridingGeneralPrices Список цен, которыми переопределять фиксированные ставки на группах
     *                                (длина списка = количество групп). Если null, не переопределять
     *                                Параметр актуален для групп типа cpm_yndx_frontpage (тип кампании соответствующий)
     *                                Для них при копировании групп автобюджетной кампании иная логика вычисления
     *                                значения
     *                                ставки в базе
     */
    private void fillDataOnCopy(ClientId clientId, List<ComplexCpmAdGroup> complexCpmAdGroups,
                                List<WebCpmAdGroup> webAdGroups, @Nullable List<Double> overridingGeneralPrices) {
        checkArgument(complexCpmAdGroups.size() == webAdGroups.size());
        List<Double> generalPrices = overridingGeneralPrices != null ? overridingGeneralPrices :
                mapList(webAdGroups, adGroup -> ifNotNull(getGeneralPrice(adGroup), BigDecimal::doubleValue));
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        List<List<Keyword>> adGroupsKeywords = mapList(complexCpmAdGroups, ComplexCpmAdGroup::getKeywords);
        copyAdGroupDataFiller.fillKeywordPrices(shard, clientId, adGroupsKeywords, generalPrices);

        List<List<TargetInterest>> adGroupsTargetInterests =
                mapList(complexCpmAdGroups, ComplexCpmAdGroup::getTargetInterests);
        copyAdGroupDataFiller.fillRetargetingPrices(shard, adGroupsTargetInterests, generalPrices);
    }

    private BigDecimal getCpmBannerDefaultPrice(Currency clientCurrency) {
        return clientCurrency.getMinCpmPrice();
    }


    private BigDecimal getCpmYndxFrontpageDefaultPrice(List<ComplexCpmAdGroup> complexCpmAdGroups,
                                                       int shard, Currency clientCurrency) {
        return cpmYndxFrontpageCurrencyService
                .getAdGroupIndexesToPriceDataMapByAdGroups(mapList(complexCpmAdGroups, ComplexAdGroup::getAdGroup),
                        shard, clientCurrency)
                .get(0)
                .getCpmYndxFrontpageMinPrice();
    }

    /**
     * Получаем список минимальных ставок для копируемых групп типа сpm_yndx_frontpage автобюджетной капмании
     *
     * @param cpmYndxFrontpageCurrencyAdGroupRestrictions мапа индексов групп в информацию с ограничениями по ним
     * @return список минимальных ставок в порядке, соответствующим индексам в
     * cpmYndxFrontpageCurrencyAdGroupRestrictions
     */
    private List<Double> getCpmYndxFrontpageGeneralPricesForCopy(
            Map<Integer, CpmYndxFrontpageAdGroupPriceRestrictions> cpmYndxFrontpageCurrencyAdGroupRestrictions) {
        List<Integer> indexList =
                cpmYndxFrontpageCurrencyAdGroupRestrictions.keySet().stream().sorted().collect(Collectors.toList());
        List<BigDecimal> minPrices = mapList(indexList,
                index -> cpmYndxFrontpageCurrencyAdGroupRestrictions.get(index).getCpmYndxFrontpageMinPrice());
        return mapList(minPrices, price -> ifNotNull(price, BigDecimal::doubleValue));
    }

    /**
     * Получение параметров для автоматического выставления недостающих ставок.
     * Если в группе не было обнаружено ставки - минимальная будет использована в качестве фиксированной.
     */
    private ShowConditionAutoPriceParams getAutoPriceParamsForAdd(List<WebCpmAdGroup> adGroups,
                                                                  BigDecimal defaultPrice) {
        ShowConditionFixedAutoPrices fixedAdGroupAutoPrices =
                ShowConditionFixedAutoPrices
                        .ofGlobalFixedPrice(Optional.ofNullable(getGeneralPrice(adGroups.get(0))).orElse(defaultPrice));
        return new ShowConditionAutoPriceParams(
                fixedAdGroupAutoPrices,
                recentStatisticsService
        );
    }


    private ShowConditionAutoPriceParams getAutoPriceParamsForCopy() {
        ShowConditionFixedAutoPrices fixedAdGroupAutoPrices = ShowConditionFixedAutoPrices.ofGlobalFixedPrice(null);
        return new ShowConditionAutoPriceParams(fixedAdGroupAutoPrices, recentStatisticsService);
    }
}
