package ru.yandex.direct.web.entity.adgroup.service.cpm;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.ComplexAdGroup;
import ru.yandex.direct.core.entity.adgroup.container.ComplexCpmAdGroup;
import ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupUpdateOperationFactory;
import ru.yandex.direct.core.entity.adgroup.service.complex.cpm.ComplexCpmAdGroupUpdateOperation;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.currency.service.CpmYndxFrontpageCurrencyService;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionAutoPriceParams;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionFixedAutoPrices;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.web.entity.adgroup.model.WebCpmAdGroup;
import ru.yandex.direct.ytcore.entity.statistics.service.RecentStatisticsService;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.web.entity.adgroup.converter.CpmAdGroupConverter.webAdGroupsToCoreComplexCpmAdGroups;
import static ru.yandex.direct.web.entity.adgroup.service.cpm.CpmAdGroupHelper.getGeneralPrice;

@Service
public class UpdateCpmAdGroupService {

    private final ComplexAdGroupUpdateOperationFactory complexAdGroupUpdateOperationFactory;
    private final ClientGeoService clientGeoService;
    private final ClientService clientService;
    private final WebCpmAdGroupValidationService webCpmAdGroupValidationService;
    private final RecentStatisticsService recentStatisticsService;
    private final ShardHelper shardHelper;
    private final CpmYndxFrontpageCurrencyService cpmYndxFrontpageCurrencyService;
    private final CpmAdGroupHelper cpmAdGroupHelper;

    public UpdateCpmAdGroupService(
            ComplexAdGroupUpdateOperationFactory complexAdGroupUpdateOperationFactory,
            ClientGeoService clientGeoService,
            ClientService clientService,
            WebCpmAdGroupValidationService webCpmAdGroupValidationService,
            RecentStatisticsService recentStatisticsService,
            ShardHelper shardHelper,
            CpmYndxFrontpageCurrencyService cpmYndxFrontpageCurrencyService,
            CpmAdGroupHelper cpmAdGroupHelper) {
        this.complexAdGroupUpdateOperationFactory = complexAdGroupUpdateOperationFactory;
        this.clientGeoService = clientGeoService;
        this.clientService = clientService;
        this.webCpmAdGroupValidationService = webCpmAdGroupValidationService;
        this.recentStatisticsService = recentStatisticsService;
        this.shardHelper = shardHelper;
        this.cpmYndxFrontpageCurrencyService = cpmYndxFrontpageCurrencyService;
        this.cpmAdGroupHelper = cpmAdGroupHelper;
    }

    public MassResult<Long> updateAdGroups(List<WebCpmAdGroup> adGroups, long campaignId, long operatorUid,
                                           ClientId clientId, long clientUid, boolean saveDraft) {
        Currency clientCurrency = clientService.getWorkCurrency(clientId);

        int shard = shardHelper.getShardByClientId(clientId);
        Campaign campaign = cpmAdGroupHelper.getCampaign(shard, campaignId);
        boolean autoBudget = campaign.getStrategy().isAutoBudget();

        var cpmYndxFrontpageCurrencyRestrictions =
                cpmAdGroupHelper.getCpmYndxFrontpagePriceRestrictions(shard, adGroups, clientCurrency, campaign);
        var validationData = new WebCpmAdGroupValidationService.ValidationData(
                clientCurrency, autoBudget, campaign.getType(), cpmYndxFrontpageCurrencyRestrictions);
        var vr = webCpmAdGroupValidationService.validate(adGroups, validationData);
        if (vr.hasAnyErrors()) {
            return MassResult.brokenMassAction(emptyList(), vr);
        }

        List<ComplexCpmAdGroup> complexAdGroups =
                webAdGroupsToCoreComplexCpmAdGroups(adGroups, campaignId, campaign.getType());

        GeoTree geoTree = clientGeoService.getClientTranslocalGeoTree(clientId);

        Map<Long, BigDecimal> defaultPrice = campaign.getType() == CampaignType.CPM_YNDX_FRONTPAGE ?
                getCpmYndxFrontpageDefaultPrice(complexAdGroups, shard, clientCurrency) :
                getCpmBannerDefaultPrices(complexAdGroups, clientCurrency);
        ShowConditionAutoPriceParams autoPriceParams = getAutoPriceParams(adGroups, defaultPrice);

        ComplexCpmAdGroupUpdateOperation updateOperation = complexAdGroupUpdateOperationFactory
                .createCpmAdGroupUpdateOperation(complexAdGroups, geoTree, true, autoPriceParams,
                        operatorUid, clientId, clientUid, saveDraft);

        return updateOperation.prepareAndApply();
    }

    private Map<Long, BigDecimal> getCpmYndxFrontpageDefaultPrice(List<ComplexCpmAdGroup> complexCpmAdGroups,
                                                                  int shard, Currency clientCurrency) {
        return EntryStream.of(
                cpmYndxFrontpageCurrencyService
                        .getAdGroupIdsToPriceDataMapByAdGroups(mapList(complexCpmAdGroups, ComplexAdGroup::getAdGroup),
                                shard, clientCurrency))
                .mapValues(t -> t.getCpmYndxFrontpageMinPrice())
                .filterValues(Objects::nonNull)
                .toMap();
    }

    private Map<Long, BigDecimal> getCpmBannerDefaultPrices(List<ComplexCpmAdGroup> complexCpmAdGroups,
                                                            Currency clientCurrency) {
        return StreamEx.of(complexCpmAdGroups)
                .toMap(t -> t.getAdGroup().getId(), t -> clientCurrency.getMinCpmPrice());
    }

    /**
     * Получение параметров для автоматического выставления недостающих ставок.
     * Если в группе не было обнаружено ставки - минимальная будет использована в качестве фиксированной.
     */
    private ShowConditionAutoPriceParams getAutoPriceParams(List<WebCpmAdGroup> adGroups,
                                                            Map<Long, BigDecimal> defaultPrices) {
        Map<Long, BigDecimal> adGroupsGeneralPrices = listToMap(adGroups, WebCpmAdGroup::getId, adGroup ->
                Optional.ofNullable(getGeneralPrice(adGroup)).orElse(defaultPrices.get(adGroup.getId())));
        ShowConditionFixedAutoPrices showConditionFixedAutoPrices =
                ShowConditionFixedAutoPrices.ofPerAdGroupFixedPrices(adGroupsGeneralPrices);
        return new ShowConditionAutoPriceParams(
                showConditionFixedAutoPrices,
                recentStatisticsService
        );
    }
}
