package ru.yandex.direct.web.entity.adgroup.service.text;

import java.math.BigDecimal;
import java.util.List;

import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.ComplexTextAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupAddOperationFactory;
import ru.yandex.direct.core.entity.adgroup.service.complex.text.ComplexTextAdGroupAddOperation;
import ru.yandex.direct.core.entity.banner.service.DatabaseMode;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.core.entity.relevancematch.model.RelevanceMatch;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionAutoPriceParams;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionFixedAutoPrices;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.entity.adgroup.model.WebTextAdGroup;
import ru.yandex.direct.web.entity.adgroup.service.CopyAdGroupDataFiller;
import ru.yandex.direct.ytcore.entity.statistics.service.RecentStatisticsService;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Collections.emptyList;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.web.entity.adgroup.converter.TextAdGroupConverter.webAdGroupsToCoreComplexTextAdGroups;

@Service
public class AddTextAdGroupService {

    private final ComplexAdGroupAddOperationFactory complexAdGroupAddOperationFactory;
    private final ClientGeoService clientGeoService;
    private final ClientService clientService;
    private final WebTextAdGroupValidationService webTextAdGroupValidationService;
    private final RecentStatisticsService recentStatisticsService;
    private final CopyAdGroupDataFiller copyAdGroupDataFiller;
    private final ShardHelper shardHelper;

    public AddTextAdGroupService(
            ComplexAdGroupAddOperationFactory complexAdGroupAddOperationFactory,
            ClientGeoService clientGeoService,
            ClientService clientService,
            WebTextAdGroupValidationService webTextAdGroupValidationService,
            RecentStatisticsService recentStatisticsService,
            CopyAdGroupDataFiller copyAdGroupDataFiller, ShardHelper shardHelper) {
        this.complexAdGroupAddOperationFactory = complexAdGroupAddOperationFactory;
        this.clientGeoService = clientGeoService;
        this.clientService = clientService;
        this.webTextAdGroupValidationService = webTextAdGroupValidationService;
        this.recentStatisticsService = recentStatisticsService;
        this.copyAdGroupDataFiller = copyAdGroupDataFiller;
        this.shardHelper = shardHelper;
    }

    public MassResult<Long> addAdGroups(List<WebTextAdGroup> adGroups, long campaignId,
                                        boolean isCopy, boolean saveDraft, long operatorUid, ClientId clientId, long clientUid) {
        Currency clientCurrency = clientService.getWorkCurrency(clientId);
        ValidationResult<List<WebTextAdGroup>, Defect> vr =
                webTextAdGroupValidationService.validate(adGroups, clientCurrency, AdGroupType.BASE);
        if (vr.hasAnyErrors()) {
            return MassResult.brokenMassAction(emptyList(), vr);
        }

        ShowConditionAutoPriceParams autoPriceParams;
        List<ComplexTextAdGroup> complexAdGroups = webAdGroupsToCoreComplexTextAdGroups(adGroups, campaignId);
        if (isCopy) {
            autoPriceParams = getAutoPriceParamsForCopy();
            fillDataOnCopy(clientId, complexAdGroups, adGroups);
        } else {
            autoPriceParams = getAutoPriceParamsForAdd(adGroups);
        }

        GeoTree geoTree = clientGeoService.getClientTranslocalGeoTree(clientId);

        ComplexTextAdGroupAddOperation addOperation = complexAdGroupAddOperationFactory
                .createTextAdGroupAddOperation(saveDraft, complexAdGroups, geoTree, true, autoPriceParams,
                        operatorUid, clientId, clientUid, DatabaseMode.ONLY_MYSQL);
        return addOperation.prepareAndApply();
    }

    private void fillDataOnCopy(ClientId clientId, List<ComplexTextAdGroup> complexTextAdGroups,
                                List<WebTextAdGroup> webTextAdGroups) {
        checkArgument(complexTextAdGroups.size() == webTextAdGroups.size());
        List<Double> generalPrices = mapList(webTextAdGroups, WebTextAdGroup::getGeneralPrice);

        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        List<List<Keyword>> adGroupsKeywords = mapList(complexTextAdGroups, ComplexTextAdGroup::getKeywords);
        copyAdGroupDataFiller.fillKeywordPrices(shard, clientId, adGroupsKeywords, generalPrices);

        List<List<TargetInterest>> adGroupsTargetInterests =
                mapList(complexTextAdGroups, ComplexTextAdGroup::getTargetInterests);
        copyAdGroupDataFiller.fillRetargetingPrices(shard, adGroupsTargetInterests, generalPrices);

        List<List<RelevanceMatch>> adGroupsRelevanceMatches =
                mapList(complexTextAdGroups, ComplexTextAdGroup::getRelevanceMatches);
        copyAdGroupDataFiller
                .fillRelevanceMatchPrices(shard, clientId, adGroupsRelevanceMatches, generalPrices);
    }

    /**
     * Получение параметров для автоматического выставления недостающих ставок.
     * <p>
     * Если в добавляемой группе есть параметр {@code generalPrice},
     * то указываем, что эту ставку нужно выставить всем условиям показов.
     * <p>
     * Тут ожидается, что добавлять можно только одну группу за запрос, т.к.
     * иначе механизм выставления фиксированных автоставок нужно переделать.
     * Поэтому этот механизм не используется при копировании групп,
     * так как за раз можно копировать несколько групп.
     */
    private ShowConditionAutoPriceParams getAutoPriceParamsForAdd(List<WebTextAdGroup> adGroups) {
        ShowConditionFixedAutoPrices fixedAdGroupAutoPrices =
                ShowConditionFixedAutoPrices.ofGlobalFixedPrice(
                        ifNotNull(adGroups.get(0).getGeneralPrice(), BigDecimal::valueOf)
                );
        return new ShowConditionAutoPriceParams(
                fixedAdGroupAutoPrices,
                recentStatisticsService
        );
    }

    private ShowConditionAutoPriceParams getAutoPriceParamsForCopy() {
        ShowConditionFixedAutoPrices fixedAdGroupAutoPrices =
                ShowConditionFixedAutoPrices.ofGlobalFixedPrice(null);
        return new ShowConditionAutoPriceParams(
                fixedAdGroupAutoPrices,
                recentStatisticsService
        );
    }
}
