package ru.yandex.direct.web.entity.adgroup.service.text;

import java.math.BigDecimal;
import java.util.List;

import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.container.ComplexTextAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.service.complex.ComplexAdGroupUpdateOperationFactory;
import ru.yandex.direct.core.entity.adgroup.service.complex.text.ComplexTextAdGroupUpdateOperation;
import ru.yandex.direct.core.entity.client.service.ClientGeoService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionAutoPriceParams;
import ru.yandex.direct.core.entity.showcondition.container.ShowConditionFixedAutoPrices;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.entity.adgroup.converter.TextAdGroupConverter;
import ru.yandex.direct.web.entity.adgroup.model.WebTextAdGroup;
import ru.yandex.direct.ytcore.entity.statistics.service.RecentStatisticsService;

import static java.util.Collections.emptyList;

@Service
public class UpdateTextAdGroupService {

    private final ComplexAdGroupUpdateOperationFactory complexAdGroupUpdateOperationFactory;
    private final ClientGeoService clientGeoService;
    private final ClientService clientService;
    private final WebTextAdGroupValidationService webTextAdGroupValidationService;
    private final RecentStatisticsService recentStatisticsService;

    public UpdateTextAdGroupService(ComplexAdGroupUpdateOperationFactory complexAdGroupUpdateOperationFactory,
                                    ClientGeoService clientGeoService,
                                    ClientService clientService,
                                    WebTextAdGroupValidationService webTextAdGroupValidationService,
                                    RecentStatisticsService recentStatisticsService) {
        this.complexAdGroupUpdateOperationFactory = complexAdGroupUpdateOperationFactory;
        this.clientGeoService = clientGeoService;
        this.clientService = clientService;
        this.webTextAdGroupValidationService = webTextAdGroupValidationService;
        this.recentStatisticsService = recentStatisticsService;
    }

    public MassResult<Long> updateAdGroups(List<WebTextAdGroup> adGroups, long campaignId, long operatorUid,
                                           ClientId clientId, long clientUid, boolean saveDraft) {
        List<ComplexTextAdGroup> complexAdGroups = TextAdGroupConverter
                .webAdGroupsToCoreComplexTextAdGroups(adGroups, campaignId);

        Currency clientCurrency = clientService.getWorkCurrency(clientId);
        ValidationResult<List<WebTextAdGroup>, Defect> vr =
                webTextAdGroupValidationService.validate(adGroups, clientCurrency, AdGroupType.BASE);
        if (vr.hasAnyErrors()) {
            return MassResult.brokenMassAction(emptyList(), vr);
        }

        GeoTree geoTree = clientGeoService.getClientTranslocalGeoTree(clientId);

        ShowConditionAutoPriceParams autoPriceParams = getAutoPriceParams(adGroups);

        ComplexTextAdGroupUpdateOperation updateOperation = complexAdGroupUpdateOperationFactory
                .createTextAdGroupUpdateOperation(complexAdGroups, geoTree, true, autoPriceParams,
                        false, operatorUid, clientId, clientUid, saveDraft);

        return updateOperation.prepareAndApply();
    }

    /**
     * Получение параметров для автоматического выставления недостающих ставок.
     * <p>
     * Указываем, что в каких-то группах нужно выставить фиксированную ставку
     * для новых/изменившихся условий показов.
     */
    private ShowConditionAutoPriceParams getAutoPriceParams(List<WebTextAdGroup> adGroups) {
        ShowConditionFixedAutoPrices showConditionFixedAutoPrices =
                ShowConditionFixedAutoPrices.ofPerAdGroupFixedPrices(
                        StreamEx.of(adGroups)
                                .filter(adGroup -> adGroup.getGeneralPrice() != null)
                                .mapToEntry(WebTextAdGroup::getId, adGroup -> BigDecimal.valueOf(adGroup.getGeneralPrice()))
                                .toMap()
                );
        return new ShowConditionAutoPriceParams(
                showConditionFixedAutoPrices,
                recentStatisticsService
        );
    }
}
