package ru.yandex.direct.web.entity.agencyofflinereport.service;

import java.time.LocalDate;
import java.util.function.Predicate;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.agencyofflinereport.service.AgencyOfflineReportParametersService;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.constraint.CommonConstraints;
import ru.yandex.direct.validation.constraint.DateConstraints;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.entity.agencyofflinereport.model.EnqueueReportRequestParams;

import static ru.yandex.direct.core.entity.agencyofflinereport.service.AgencyOfflineReportParametersService.MIN_REPORT_DATE;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;

@Service
public class AgencyOfflineReportWebValidationService {

    private final AgencyOfflineReportParametersService parametersService;

    @Autowired
    public AgencyOfflineReportWebValidationService(AgencyOfflineReportParametersService parametersService) {
        this.parametersService = parametersService;
    }

    private static final Predicate<EnqueueReportRequestParams> END_DATE_NOT_BEFORE_START =
            p -> p.getDateTo().compareTo(p.getDateFrom()) >= 0;

    private static final Constraint<Long, Defect> ID_VALIDATOR = CommonConstraints.validId();

    public ValidationResult<EnqueueReportRequestParams, Defect> validateEnqueueRequest(
            EnqueueReportRequestParams params) {
        LocalDate maxReportDate = parametersService.getMaximumAvailableDate();

        ItemValidationBuilder<EnqueueReportRequestParams, Defect> vb = ItemValidationBuilder.of(params);
        vb.check(notNull());

        vb.item(params.getDateFrom(), EnqueueReportRequestParams.DATE_FROM_FIELD_NAME)
                .check(notNull())
                .check(DateConstraints.isNotBeforeThan(MIN_REPORT_DATE), When.notNull())
                .check(DateConstraints.isNotAfterThan(maxReportDate), When.notNull());

        vb.item(params.getDateTo(), EnqueueReportRequestParams.DATE_TO_FIELD_NAME)
                .check(notNull())
                .check(DateConstraints.isNotAfterThan(maxReportDate), When.notNull());

        vb.check(Constraint.fromPredicate(END_DATE_NOT_BEFORE_START, CommonDefects.inconsistentState()), When.isValid());

        return vb.getResult();
    }

    public boolean isValidReportId(Long reportId) {
        return ID_VALIDATOR.apply(reportId) == null;
    }
}
