package ru.yandex.direct.web.entity.banner.converter;

import java.math.BigDecimal;
import java.util.List;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.banner.container.ComplexBanner;
import ru.yandex.direct.core.entity.banner.model.BannerPrice;
import ru.yandex.direct.core.entity.banner.model.BannerPricesCurrency;
import ru.yandex.direct.core.entity.banner.model.BannerPricesPrefix;
import ru.yandex.direct.core.entity.banner.model.BannerWithHref;
import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.banner.model.BannerWithTurboLanding;
import ru.yandex.direct.core.entity.banner.model.BannerWithTurboLandingParams;
import ru.yandex.direct.core.entity.banner.model.CpcVideoBanner;
import ru.yandex.direct.core.entity.banner.model.CpmAudioBanner;
import ru.yandex.direct.core.entity.banner.model.ImageBanner;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.dbschema.ppc.enums.BannersBannerType;
import ru.yandex.direct.web.entity.banner.model.WebBanner;
import ru.yandex.direct.web.entity.banner.model.WebBannerCallout;
import ru.yandex.direct.web.entity.banner.model.WebBannerCreative;
import ru.yandex.direct.web.entity.banner.model.WebBannerImageAd;
import ru.yandex.direct.web.entity.banner.model.WebBannerPrice;
import ru.yandex.direct.web.entity.banner.model.WebBannerTurbolanding;

import static com.google.common.base.Preconditions.checkArgument;
import static java.lang.String.format;
import static java.math.RoundingMode.HALF_UP;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static ru.yandex.direct.core.entity.domain.DomainUtils.refineDomain;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.NumberUtils.nullIfZero;
import static ru.yandex.direct.utils.StringUtils.ifNotBlank;
import static ru.yandex.direct.utils.StringUtils.nullIfBlank;
import static ru.yandex.direct.web.entity.banner.converter.BannerSitelinkConverter.toCoreSitelinks;
import static ru.yandex.direct.web.entity.banner.converter.BannerVcardConverter.toCoreVcard;

@ParametersAreNonnullByDefault
public class BannerConverter {

    private BannerConverter() {
    }

    public static List<ComplexBanner> webBannersToCoreComplexBanners(@Nullable List<WebBanner> webBanners) {
        return mapList(webBanners, BannerConverter::webBannerToCoreComplexBanner);
    }

    @Nullable
    static ComplexBanner webBannerToCoreComplexBanner(@Nullable WebBanner webBanner) {
        return webBanner == null ? null : new ComplexBanner()
                .withBanner(webBannerToCoreBanner(webBanner))
                .withVcard(toCoreVcard(webBanner.getVcard()))
                .withSitelinkSet(toCoreSitelinks(webBanner.getSitelinks()));
    }

    @Nullable
    private static BannerWithSystemFields webBannerToCoreBanner(@Nullable WebBanner webBanner) {
        if (webBanner == null) {
            return null;
        }
        final BannerWithSystemFields banner;
        String adType = webBanner.getAdType();

        checkArgument(adType != null, "AdType must be non-null");

        if (BannersBannerType.image_ad.getLiteral().equals(adType)) {

            if (webBanner.getCreative() == null || isBlank(webBanner.getCreative().getCreativeId())) {  // Smart.pm:1065
                banner = new ImageBanner()
                        .withIsMobileImage(false)
                        .withImageHash(ifNotNull(webBanner.getImageAd(), WebBannerImageAd::getHash));
            } else {
                String creativeId = ifNotNull(webBanner.getCreative(), WebBannerCreative::getCreativeId);
                banner = new ImageBanner()
                        .withIsMobileImage(false)
                        .withCreativeId(ifNotBlank(creativeId, Long::parseLong));
            }

        } else if (BannersBannerType.text.getLiteral().equals(adType)) {
            List<Long> calloutIds = ifNotNull(webBanner.getCallouts(),
                    callouts -> mapList(callouts, WebBannerCallout::getAdditionsItemId));
            Long creativeId = ifNotNull(webBanner.getVideoResources(),
                    videoRes -> nullIfZero(videoRes.getId()));
            boolean isMobile = "mobile".equals(webBanner.getBannerType());
            banner = new TextBanner()
                    .withIsMobile(isMobile)
                    .withTitle(webBanner.getTitle())
                    .withTitleExtension(nullIfBlank(webBanner.getTitleExtension()))
                    .withBody(webBanner.getBody())
                    .withDisplayHref(nullIfBlank(webBanner.getDisplayHref()))
                    .withImageHash(ifNotBlank(webBanner.getImageHash(), Function.identity()))
                    .withCalloutIds(calloutIds)
                    .withCreativeId(creativeId)
                    .withBannerPrice(toCoreBannerPrice(webBanner.getBannerPrice()))
                    .withPermalinkId(webBanner.getPermalinkId());
        } else if (BannersBannerType.cpc_video.getLiteral().equals(adType)) {
            String creativeId = ifNotNull(webBanner.getCreative(), WebBannerCreative::getCreativeId);

            banner = new CpcVideoBanner()
                    .withIsMobileVideo(false)
                    .withCreativeId(ifNotBlank(creativeId, Long::parseLong));
        } else if (BannersBannerType.cpm_audio.getLiteral().equals(adType)) {
            String creativeId = ifNotNull(webBanner.getCreative(), WebBannerCreative::getCreativeId);

            banner = new CpmAudioBanner()
                    .withCreativeId(ifNotBlank(creativeId, Long::parseLong));
        } else {
            throw new UnsupportedOperationException(format("Ads of type %s are not supported.", adType));
        }

        String href = !isBlank(webBanner.getUrlProtocol()) && !isBlank(webBanner.getHref()) ?
                webBanner.getUrlProtocol() + webBanner.getHref() :
                null;

        // Модель WebBannerTurbolanding используется только в старом интерфейсе,
        // в новом следует использовать turbolandingId
        //noinspection ConstantConditions
        if (banner instanceof BannerWithTurboLanding) {
            ((BannerWithTurboLanding) banner)
                    .withTurboLandingId(toCoreTurbolandingId(webBanner.getTurbolanding()));
        }
        //noinspection ConstantConditions
        if (banner instanceof BannerWithTurboLandingParams) {
            var hrefParams = webBanner.getTurbolandingHrefParams();
            ((BannerWithTurboLandingParams) banner)
                    .withTurboLandingHrefParams(isBlank(hrefParams) ? null : hrefParams);
        }
        //noinspection ConstantConditions
        if (banner instanceof BannerWithHref) {
            ((BannerWithHref) banner)
                    .withHref(href)
                    .withDomain(refineDomain(nullIfBlank(webBanner.getDomain())));
        }

        return banner
                .withId(nullIfZero(webBanner.getId()));
    }

    static Long toCoreTurbolandingId(@Nullable WebBannerTurbolanding webTurbo) {
        return webTurbo == null ? null : nullIfZero(webTurbo.getId());
    }

    @Nullable
    static BannerPrice toCoreBannerPrice(@Nullable WebBannerPrice webBannerPrice) {
        if (webBannerPrice == null || isBlank(webBannerPrice.getPrice())) {
            return null;
        }
        return new BannerPrice()
                .withPrefix(ifNotBlank(webBannerPrice.getPrefix(),
                        p -> BannerPricesPrefix.valueOf(p.toUpperCase())))
                .withCurrency(BannerPricesCurrency.valueOf(webBannerPrice.getCurrency().toUpperCase()))
                .withPrice(webPriceToCorePrice(webBannerPrice.getPrice()))
                .withPriceOld(ifNotBlank(webBannerPrice.getPriceOld(), BannerConverter::webPriceToCorePrice));
    }

    @Nonnull
    private static BigDecimal webPriceToCorePrice(@Nonnull String webPrice) {
        return new BigDecimal(webPrice.replaceAll(" ", "").replaceAll(",", "."))
                .setScale(2, HALF_UP);
    }
}
