package ru.yandex.direct.web.entity.banner.service;

import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.hrefparams.service.HrefWithParamsBuildingService;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.web.core.model.WebErrorResponse;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;
import ru.yandex.direct.web.entity.banner.model.BannerHrefWithParamsResponse;

import static org.apache.commons.lang3.StringUtils.isBlank;
import static ru.yandex.direct.validation.constraint.StringConstraints.isValidHref;

@Service
@ParametersAreNonnullByDefault
public class BannerHrefWithParamsWebService {
    public static final String INVALID_HREF_MESSAGE = "No valid href for banner found";
    public static final String INVALID_UID_OR_ADGROUP_ID_MESSAGE = "Ad group id is invalid";

    private final ShardHelper shardHelper;
    private final DirectWebAuthenticationSource authenticationSource;
    private final BannerCommonRepository bannerRepository;
    private final HrefWithParamsBuildingService hrefWithParamsBuildingService;
    private final AdGroupRepository adGroupRepository;

    @Autowired
    public BannerHrefWithParamsWebService(ShardHelper shardHelper,
                                          DirectWebAuthenticationSource authenticationSource,
                                          BannerCommonRepository bannerRepository,
                                          HrefWithParamsBuildingService hrefWithParamsBuildingService,
                                          AdGroupRepository adGroupRepository) {
        this.shardHelper = shardHelper;
        this.authenticationSource = authenticationSource;
        this.bannerRepository = bannerRepository;
        this.hrefWithParamsBuildingService = hrefWithParamsBuildingService;
        this.adGroupRepository = adGroupRepository;
    }

    public WebResponse getBannerHrefWithParams(Long adGroupId,
                                               @Nullable Long bannerId,
                                               @Nullable String bannerHref) {
        var clientId = authenticationSource.getAuthentication().getSubjectUser().getClientId();
        int shard = shardHelper.getShardByClientId(clientId);

        var adGroupExists = adGroupRepository.getClientExistingAdGroupIds(shard, clientId, List.of(adGroupId))
                .contains(adGroupId);
        if (!adGroupExists) {
            return new WebErrorResponse(HttpStatus.BAD_REQUEST.value(), INVALID_UID_OR_ADGROUP_ID_MESSAGE);
        }

        String currentBannerHref = extractBannerHref(shard, bannerId, bannerHref);
        if (currentBannerHref == null) {
            return new WebErrorResponse(HttpStatus.BAD_REQUEST.value(), INVALID_HREF_MESSAGE);
        }

        return new BannerHrefWithParamsResponse(
                hrefWithParamsBuildingService.buildHrefWithParamsByAdGroupId(shard, adGroupId, currentBannerHref)
        );
    }

    /**
     * Если в запросе непустой href, работаем с ним. Иначе достаем href из базы.
     * Если href невалиден или его не удалось найти в базе, возвращается null.
     */
    private String extractBannerHref(int shard,
                                     @Nullable Long bannerId,
                                     @Nullable String bannerHref) {
        if (isBlank(bannerHref)) {
            return bannerRepository.getHref(shard, bannerId);
        }
        return isValidHref(bannerHref) ? bannerHref : null;
    }
}
