package ru.yandex.direct.web.entity.banner.service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.IntStreamEx;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import ru.yandex.direct.core.entity.image.container.BannerImageType;
import ru.yandex.direct.core.entity.image.container.UploadedBannerImageInformation;
import ru.yandex.direct.core.entity.image.model.BannerImageSource;
import ru.yandex.direct.core.entity.image.model.ImageUploadContainer;
import ru.yandex.direct.core.entity.image.service.ImageService;
import ru.yandex.direct.core.entity.internalads.model.InternalTemplateInfo;
import ru.yandex.direct.core.entity.internalads.model.ResourceInfo;
import ru.yandex.direct.core.entity.internalads.model.ResourceType;
import ru.yandex.direct.core.entity.internalads.service.TemplateInfoService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.core.entity.banner.service.validation.BannerImageValidationService;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;
import ru.yandex.direct.web.entity.banner.model.BannerImageUploadResponse;
import ru.yandex.direct.web.entity.banner.model.BannerImagesUploadResponse;
import ru.yandex.direct.web.validation.kernel.ValidationResultConversionService;

import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;
import static ru.yandex.direct.operation.tree.TreeOperationUtils.mergeSubListValidationResults;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.web.core.entity.banner.converter.BannerImageConverter.toWebUploadedImage;
import static ru.yandex.direct.web.core.entity.banner.converter.BannerImageConverter.toWebUploadedImages;

@Service
@ParametersAreNonnullByDefault
public class BannerImageWebService {
    private final DirectWebAuthenticationSource authenticationSource;
    private final BannerImageValidationService bannerImageValidationService;
    private final ValidationResultConversionService validationResultConversionService;
    private final ImageService bannerImageService;
    private final TemplateInfoService templateInfoService;

    public BannerImageWebService(
            DirectWebAuthenticationSource authenticationSource,
            BannerImageValidationService bannerImageValidationService,
            ValidationResultConversionService validationResultConversionService,
            ImageService bannerImageService,
            TemplateInfoService templateInfoService) {
        this.authenticationSource = authenticationSource;
        this.bannerImageValidationService = bannerImageValidationService;
        this.validationResultConversionService = validationResultConversionService;
        this.bannerImageService = bannerImageService;
        this.templateInfoService = templateInfoService;
    }

    public WebResponse saveImages(List<MultipartFile> imageFiles, String imageTypeStr) {
        var imageType = BannerImageType.valueOf(imageTypeStr.toUpperCase());

        return saveImages(imageType, false, () -> createUploadContainers(imageFiles));
    }

    public WebResponse saveImagesForInternalAd(List<Long> templateIds,
                                               List<Long> resourceIds,
                                               List<MultipartFile> imageFiles) {
        if (templateIds.size() != resourceIds.size() || templateIds.size() != imageFiles.size()) {
            ValidationResult<Object, Defect> vr = ValidationResult.failed(null, CommonDefects.invalidFormat());
            return validationResultConversionService.buildValidationResponse(vr);
        }

        BannerImageType imageType = BannerImageType.BANNER_INTERNAL;

        return saveImages(imageType, true, () -> createUploadContainers(imageFiles, templateIds, resourceIds));
    }

    private WebResponse saveImages(BannerImageType imageType,
                                   boolean validationForInternalAd,
                                   Supplier<Map<Integer, ImageUploadContainer>> fileContainersSupplier) {
        var fileContainersById = fileContainersSupplier.get();

        ValidationResult<List<Integer>, Defect> validation = bannerImageValidationService.validateImages(
                new ArrayList<>(fileContainersById.keySet()),
                fileContainersById,
                imageType,
                validationForInternalAd
        );
        Map<Integer, Integer> validItemsByIndex = ValidationResult.getValidItemsWithIndex(validation);
        if (validItemsByIndex.size() == 0) {
            return validationResultConversionService.buildValidationResponse(validation);
        }

        ClientId clientId = authenticationSource.getAuthentication().getSubjectUser().getClientId();
        var validImageFileByIndex =
                listToMap(validItemsByIndex.values(), Function.identity(), fileContainersById::get);
        MassResult<UploadedBannerImageInformation> operationResult =
                bannerImageService.saveImages(clientId, validImageFileByIndex, imageType);

        List<Integer> indexes = new ArrayList<>(validItemsByIndex.keySet());
        Map<Integer, Integer> destToSourceIndexMap = EntryStream.of(indexes)
                .invert()
                .toMap();
        //noinspection unchecked
        mergeSubListValidationResults(validation,
                (ValidationResult<List<Integer>, Defect>) operationResult.getValidationResult(),
                destToSourceIndexMap);
        if (operationResult.getSuccessfulObjectsCount() == 0) {
            return validationResultConversionService.buildValidationResponse(validation);
        }

        List<UploadedBannerImageInformation> uploadedTextBannerImageInformationList =
                StreamEx.of(operationResult.getResult())
                        .map(Result::getResult)
                        .nonNull()
                        .toList();

        return new BannerImagesUploadResponse()
                .withResult(toWebUploadedImages(uploadedTextBannerImageInformationList))
                .withValidation(validationResultConversionService
                        .buildWebValidationResult(validation));
    }

    public WebResponse saveImage(String url, String imageTypeStr) {
        BannerImageType imageType = BannerImageType.valueOf(imageTypeStr.toUpperCase());

        return saveImage(url, imageType, null);
    }

    public WebResponse saveImage(String url, Long templateId, Long resourceId) {
        var resourceInfo = getResourceFromTemplate(templateInfoService.getByTemplateId(templateId), resourceId);
        BannerImageType imageType = BannerImageType.BANNER_INTERNAL;

        return saveImage(url, imageType, resourceInfo);
    }

    private WebResponse saveImage(String url, BannerImageType imageType, @Nullable ResourceInfo resourceInfo) {
        ValidationResult<String, Defect> validation =
                bannerImageValidationService.validateImageUrl(url);

        if (validation.hasAnyErrors()) {
            return validationResultConversionService.buildValidationResponse(validation);
        }

        ClientId clientId = authenticationSource.getAuthentication().getSubjectUser().getClientId();

        @SuppressWarnings("ConstantConditions")
        Result<UploadedBannerImageInformation> operationResult =
                bannerImageService.saveImageFromUrl(clientId, url, imageType, BannerImageSource.DIRECT, resourceInfo);
        if (hasValidationIssues(operationResult)) {
            return validationResultConversionService.buildValidationResponse(operationResult);
        }
        return new BannerImageUploadResponse()
                .withResult(toWebUploadedImage(operationResult.getResult()));
    }

    private Map<Integer, ImageUploadContainer> createUploadContainers(List<MultipartFile> imageFiles) {
        return IntStreamEx.range(imageFiles.size())
                .boxed()
                .mapToEntry(i -> new ImageUploadContainer(i, imageFiles.get(i), null))
                .toMap();
    }

    private Map<Integer, ImageUploadContainer> createUploadContainers(List<MultipartFile> imageFiles,
                                                                      List<Long> templateIds,
                                                                      List<Long> resourceIds) {

        var templatesById = listToMap(
                templateInfoService.getByTemplateIds(templateIds), InternalTemplateInfo::getTemplateId
        );

        return IntStreamEx.range(imageFiles.size())
                .boxed()
                .mapToEntry(i -> new ImageUploadContainer(
                                i,
                                imageFiles.get(i),
                                getResourceFromTemplate(templatesById.get(templateIds.get(i)), resourceIds.get(i))
                        )
                ).toMap();
    }

    @Nullable
    private ResourceInfo getResourceFromTemplate(@Nullable InternalTemplateInfo templateInfo, Long resourceId) {
        if (templateInfo == null) {
            return null;
        }

        return templateInfo.getResources().stream()
                .filter(resource -> resource.getId().equals(resourceId) && resource.getType() == ResourceType.IMAGE)
                .findFirst()
                .orElse(null);
    }
}
