package ru.yandex.direct.web.entity.bannerstorage.controller

import com.fasterxml.jackson.annotation.JsonProperty
import io.swagger.annotations.Api
import io.swagger.annotations.ApiOperation
import io.swagger.annotations.ApiResponse
import io.swagger.annotations.ApiResponses
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.http.MediaType
import org.springframework.stereotype.Controller
import org.springframework.web.bind.annotation.RequestMapping
import org.springframework.web.bind.annotation.RequestMethod
import org.springframework.web.bind.annotation.RequestParam
import org.springframework.web.bind.annotation.ResponseBody
import org.springframework.web.multipart.MultipartFile
import ru.yandex.direct.bannerstorage.client.BannerStorageClient
import ru.yandex.direct.bannerstorage.client.BannerStorageClient.MAX_FILE_SIZE
import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite
import ru.yandex.direct.web.core.model.WebErrorResponse
import ru.yandex.direct.web.core.model.WebResponse
import ru.yandex.direct.web.core.security.authentication.DirectCookieAuthProvider
import ru.yandex.direct.web.core.security.captcha.DisableAutoCaptcha
import ru.yandex.direct.web.entity.MassResponse
import ru.yandex.direct.web.entity.banner.model.BannerImagesUploadResponse
import ru.yandex.direct.web.validation.model.WebDefect
import ru.yandex.direct.web.validation.model.WebValidationResult
import java.io.File

class WebUploadedFilesResponse : MassResponse<List<WebUploadedFile>>()

class WebUploadedFile(
    // id файла в bannerstorage
    @JsonProperty("id") val id: Int,
    // имя оригинального (загружаемого пользователем) файла
    @JsonProperty("name") val name: String,
    // ссылка в MDS
    @JsonProperty("url") val url: String,
    // ширина изображения/видео
    @JsonProperty("width") val width: Int?,
    // высота изображения/видео
    @JsonProperty("height") val height: Int?
) {
    override fun toString(): String {
        return "WebUploadedFile(id=$id, name=$name, url=$url)"
    }
}

@Controller
@RequestMapping(value = ["/bannerstorage"], produces = [MediaType.APPLICATION_JSON_UTF8_VALUE])
@Api(tags = ["bannerstorage"])
open class FileController @Autowired constructor(
    private val bannerStorageClient: BannerStorageClient
) {
    @ApiOperation(value = "uploadFile", httpMethod = "POST", nickname = "uploadFile")
    @ApiResponses(
        ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse::class),
        ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse::class),
        ApiResponse(code = 200, message = "Ok", response = BannerImagesUploadResponse::class)
    )
    @PreAuthorizeWrite
    @DisableAutoCaptcha
    @RequestMapping(path = ["file/upload"], method = [RequestMethod.POST], consumes = [MediaType.MULTIPART_FORM_DATA_VALUE])
    @ResponseBody
    open fun uploadFile(@RequestParam("files") files: List<MultipartFile>,
                        @RequestParam(value = DirectCookieAuthProvider.PARAMETER_ULOGIN) subjectLogin: String): WebResponse {

        if (files.isEmpty()) {
            return WebUploadedFilesResponse()
                .withValidation(WebValidationResult().addErrors(WebDefect().withText("No files specified")))
        }

        for (file in files) {
            if (file.size == 0L) {
                return WebUploadedFilesResponse()
                    .withValidation(WebValidationResult().addErrors(
                        WebDefect().withText("Provided file ${file.originalFilename} is empty")))
            }
            if (file.size > MAX_FILE_SIZE) {
                return WebUploadedFilesResponse()
                    .withValidation(WebValidationResult().addErrors(
                        WebDefect().withText("File size limit $MAX_FILE_SIZE exceeded (provided ${file.size})")))
            }
        }

        val uploadedFilesList = ArrayList<WebUploadedFile>()
        for (file in files) {
            var tempFile: File? = null
            try {
                // сохраняем stream во временный файл, чтобы не хранить файл (который может быть большим) в памяти
                tempFile = File.createTempFile("upload-", ".tmp")
                file.transferTo(tempFile!!)

                // bannerstorage не возвращает сразу url в MDS для загружаемых файлов,
                // поэтому приходится дёргать две ручки
                val uploadedFile = bannerStorageClient.uploadFile(tempFile, file.originalFilename)
                val uploadedFileExt = bannerStorageClient.getFile(uploadedFile.id)

                uploadedFilesList.add(
                    WebUploadedFile(
                        id = uploadedFile.id,
                        name = uploadedFile.name,
                        url = uploadedFileExt.stillageFileUrl,
                        width = uploadedFileExt.width,
                        height = uploadedFileExt.height))
            } finally {
                tempFile?.delete()
            }
        }

        return WebUploadedFilesResponse().withResult(uploadedFilesList)
    }
}
