package ru.yandex.direct.web.entity.bidmodifier.converter;

import java.util.Map;
import java.util.function.Function;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.bidmodifier.BidModifierExpression;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierExpressionAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifier.model.BidModifierExpressionLiteral;
import ru.yandex.direct.core.entity.bidmodifiers.Constants;
import ru.yandex.direct.core.entity.bidmodifiers.expression.ParameterType;
import ru.yandex.direct.core.entity.bidmodifiers.service.BidModifierExpressionFactory;
import ru.yandex.direct.web.core.exception.WebBadRequestException;
import ru.yandex.direct.web.entity.bidmodifier.model.WebExpressionBidModifier;
import ru.yandex.direct.web.entity.bidmodifier.model.WebExpressionBidModifierAdjustment;
import ru.yandex.direct.web.entity.bidmodifier.model.WebExpressionBidModifierLiteral;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Конвертеры универсальных корректировок из web-api моделей в ядровые.
 * <p>
 * Строки в енумы (например BidModifierType) преобразуются путём приведения строки в верхний регистр.
 * Если не удалось распарсить енум или числовое значение параметра, конвертеры кидают исключение
 * {@code WebBadRequestException}, чтобы приложение ответило 400-м кодом.
 */
public class ExpressionBidModifierConverter {

    // мапы для приведения lowercase строки значениям енумов
    private static Map<String, BidModifierType> defaultStringToTypeMap = lcToEnum(BidModifierType.values());

    private ExpressionBidModifierConverter() {
    }

    private static <T extends Enum> Map<String, T> lcToEnum(T[] values) {
        return StreamEx.of(values).mapToEntry(t -> t.toString().toLowerCase(), Function.identity()).toImmutableMap();
    }

    public static BidModifierExpression webExpressionBidModifierToCore(WebExpressionBidModifier bme) {
        BidModifierType type = parseBidModifierType(bme.getType());
        BidModifierExpression bidModifier = BidModifierExpressionFactory.createBidModifierByType(type);
        return bidModifier
                .withType(type)
                .withEnabled(bme.getEnabled() != 0)
                .withExpressionAdjustments(
                        mapList(bme.getAdjustments(), adj -> webExpressionBidModifierAdjustmentToCore(adj, type))
                );
    }


    private static BidModifierExpressionAdjustment webExpressionBidModifierAdjustmentToCore(
            WebExpressionBidModifierAdjustment s,
            BidModifierType type
    ) {
        BidModifierExpressionAdjustment adj = BidModifierExpressionFactory.createBidModifierAdjustmentByType(type);
        return adj
                .withPercent(s.getPercent())
                .withCondition(
                        mapList(s.getCondition(),
                                exp -> mapList(
                                        exp,
                                        lit -> webExpressionBidModifierLiteralToCore(lit, type)
                                ))
                );
    }

    private static BidModifierExpressionLiteral webExpressionBidModifierLiteralToCore(
            WebExpressionBidModifierLiteral lit,
            BidModifierType type) {
        var param = lit.getParameter();
        BidModifierExpressionLiteral literal = new BidModifierExpressionLiteral()
                .withParameter(param)
                .withOperation(lit.getOperation());
        var parameterInfo = Constants.ALL_PARAMETERS.get(type).get(param);
        checkNotNull(parameterInfo);
        if (parameterInfo.getType() == ParameterType.INTEGER) {
            String value = lit.getValue();
            try {
                literal.setValueInteger(Integer.parseInt(value));
            } catch (NumberFormatException e) {
                throw new WebBadRequestException("Failed to parse %s as integer", value);
            }
        } else {
            literal.setValueString(lit.getValue());
        }

        return literal;
    }

    private static BidModifierType parseBidModifierType(String type) {
        if (!defaultStringToTypeMap.containsKey(type)) {
            throw new WebBadRequestException("Unknown BidModifier type '%s'", type);
        }
        BidModifierType coreType = defaultStringToTypeMap.get(type);
        if (BidModifierExpressionFactory.notExpressionBidModifierType(coreType)) {
            throw new WebBadRequestException("BidModifier type %s is not an expression bid modifier type", type);
        }
        return coreType;
    }
}
