package ru.yandex.direct.web.entity.campaign.service;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.BrandSurveyStatus;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignSimple;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.SurveyStatus;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.CampaignBudgetReachService;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.core.entity.deal.service.DealService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.security.DirectAuthentication;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.core.model.WebErrorResponse;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.model.WebSuccessResponse;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;
import ru.yandex.direct.web.entity.campaign.model.BrandSurveyStatusWeb;
import ru.yandex.direct.web.entity.campaign.model.BrandSurveyStopReasonWeb;
import ru.yandex.direct.web.entity.campaign.model.CampaignBrandSurveyWebResponse;
import ru.yandex.direct.web.entity.campaign.model.CampaignWithDealsWeb;
import ru.yandex.direct.web.entity.campaign.model.CampaignsBrandSurveyResponse;
import ru.yandex.direct.web.entity.campaign.model.GetCampaignExtendedTypeResponse;
import ru.yandex.direct.web.entity.campaign.model.GetCampaignTypeResponse;
import ru.yandex.direct.web.entity.campaign.model.SearchDealCampaignsResponse;
import ru.yandex.direct.web.entity.campaign.model.SurveyStatusWeb;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.campaign.model.CampaignOpts.IS_BRAND_LIFT_HIDDEN;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@Service
public class WebCampaignService {

    private static final Set<CampaignType> brandLiftCampaignTypes =
            Set.of(CampaignType.CPM_BANNER, CampaignType.CPM_YNDX_FRONTPAGE);

    private final DirectWebAuthenticationSource authenticationSource;
    private final DealService dealService;
    private final ShardHelper shardHelper;
    private final CampaignBudgetReachService campaignBudgetReachService;
    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;
    private final CampaignService campaignService;
    private final CampaignRepository campaignRepository;
    private final FeatureService featureService;
    private final CampaignGeneratedFlagService campaignGeneratedFlagService;

    public WebCampaignService(
            DirectWebAuthenticationSource authenticationSource,
            DealService dealService, ShardHelper shardHelper,
            CampaignBudgetReachService campaignBudgetReachService,
            CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory,
            CampaignService campaignService,
            CampaignRepository campaignRepository,
            FeatureService featureService,
            CampaignGeneratedFlagService campaignGeneratedFlagService) {
        this.authenticationSource = authenticationSource;
        this.dealService = dealService;
        this.shardHelper = shardHelper;
        this.campaignBudgetReachService = campaignBudgetReachService;
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
        this.campaignService = campaignService;
        this.campaignRepository = campaignRepository;
        this.featureService = featureService;
        this.campaignGeneratedFlagService = campaignGeneratedFlagService;
    }

    public SearchDealCampaignsResponse searchDealCampaigns() {
        DirectAuthentication authentication = authenticationSource.getAuthentication();
        ClientId clientId = authentication.getSubjectUser().getClientId();
        List<CampaignSimple> campaigns = dealService.searchDealCampaignsByClientId(clientId);
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        Map<Long, List<Long>> linkedDeals = dealService.getLinkedDealsByCampaignsIdsInShard(shard,
                campaigns.stream().map(CampaignSimple::getId).collect(toList()));
        return new SearchDealCampaignsResponse().withResult(
                mapList(campaigns, c -> new CampaignWithDealsWeb()
                        .withId(c.getId())
                        .withName(c.getName())
                        .withStatusModerate(c.getStatusModerate().toString())
                        .withDealIds(linkedDeals.getOrDefault(c.getId(), Collections.emptyList()))));
    }

    public CampaignsBrandSurveyResponse getBrandLift(List<Long> cids, Long operatorUid, ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        CampaignSubObjectAccessValidator checker = campaignSubObjectAccessCheckerFactory
                .newCampaignChecker(operatorUid, clientId, cids)
                .createValidator(CampaignAccessType.READ);

        List<Campaign> campaigns = campaignService.getCampaigns(clientId, cids);
        Map<Long, Campaign> cidToCampaign = listToMap(campaigns, Campaign::getId, Function.identity());
        Map<Long, CampaignType> cidToCampaignType = listToMap(campaigns,
                Campaign::getId, Campaign::getType);

        boolean isBrandLiftHiddenEnabled = featureService.isEnabled(operatorUid, FeatureName.BRAND_LIFT_HIDDEN);

        List<Long> validCids = StreamEx.of(cids).map(checker)
                .filter(vr -> !vr.hasAnyErrors() && brandLiftCampaignTypes.contains(cidToCampaignType.get(vr.getValue())))
                .map(ValidationResult::getValue)
                .filter(cid -> isBrandLiftHiddenEnabled || cidToCampaign.containsKey(cid) &&
                        !cidToCampaign.get(cid).getOpts().contains(IS_BRAND_LIFT_HIDDEN))
                .toList();

        Map<Long, String> campaignIdToBrandSurveyId =
                campaignRepository.getBrandSurveyIdsForCampaigns(shard, validCids);
        Map<Long, BrandSurveyStatus> campaignIdToBrandSurveyStatus =
                campaignBudgetReachService.getBrandStatusForCampaigns(shard, clientId, campaignIdToBrandSurveyId);

        var campaignBrandSurveyResponses = mapList(campaigns,
                campaign -> getBrandSurveyStatus(
                        campaignIdToBrandSurveyId.get(campaign.getId()),
                        campaignIdToBrandSurveyStatus.get(campaign.getId())
                )
        );

        return new CampaignsBrandSurveyResponse().withResult(campaignBrandSurveyResponses);
    }

    public WebResponse deleteBrandLift(List<Long> cids, Long operatorUid, ClientId clientId) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        var validator = campaignSubObjectAccessCheckerFactory
                .newCampaignChecker(operatorUid, clientId, cids)
                .createValidator(CampaignAccessType.READ_WRITE);

        var validCids = StreamEx.of(cids)
                .map(validator)
                .filter(vr -> !vr.hasAnyErrors())
                .map(ValidationResult::getValue)
                .toList();

        var brandStatusForCampaigns = campaignBudgetReachService.getBrandStatusForCampaigns(shard, clientId, validCids);
        var cidsForBrandLiftDeletion = validCids
                .stream()
                .filter(cid -> brandStatusForCampaigns.get(cid) != null
                        && brandStatusForCampaigns.get(cid).getSurveyStatusDaily() == SurveyStatus.DRAFT)
                .collect(toList());

        campaignRepository.deleteBrandSurveyId(shard, cidsForBrandLiftDeletion);
        return new WebSuccessResponse();
    }

    public WebResponse getCampaignType(Long campaignId, Long operatorUid, ClientId clientId) {
        List<Long> campaignIds = List.of(campaignId);
        Defect defect = campaignSubObjectAccessCheckerFactory
                .newCampaignChecker(operatorUid, clientId, campaignIds)
                .createValidator(CampaignAccessType.READ)
                .getAccessConstraint()
                .apply(campaignId);
        if (defect != null) {
            return new WebErrorResponse(400, "Campaign not found");
        }

        Map<Long, CampaignType> campaignsTypes = campaignService.getCampaignsTypes(campaignIds);
        if (!campaignsTypes.containsKey(campaignId)) {
            return new WebErrorResponse(400, "Campaign not found");
        }
        return new GetCampaignTypeResponse().withResult(campaignsTypes.get(campaignId));
    }

    public WebResponse getCampaignExtendedType(Long campaignId, Long operatorUid, ClientId clientId) {
        Defect defect = campaignSubObjectAccessCheckerFactory
                .newCampaignChecker(operatorUid, clientId, List.of(campaignId))
                .createValidator(CampaignAccessType.READ)
                .getAccessConstraint()
                .apply(campaignId);
        if (defect != null) {
            return new WebErrorResponse(400, "Campaign not found");
        }

        var typeSource = campaignService
                .getCampaignsTypeSourceMap(List.of(campaignId), true)
                .get(campaignId);
        if (typeSource == null) {
            return new WebErrorResponse(400, "Campaign not found");
        }

        boolean isGenerated = campaignGeneratedFlagService.isGenerated(typeSource.getCampaignsSource(), campaignId);

        return new GetCampaignExtendedTypeResponse(
                typeSource.getCampaignType(),
                typeSource.getCampaignsSource(),
                isGenerated
        );
    }

    private CampaignBrandSurveyWebResponse getBrandSurveyStatus(
            String brandSurveyId,
            BrandSurveyStatus brandSurveyStatus
    ) {
        if (brandSurveyId == null) {
            return new CampaignBrandSurveyWebResponse();
        }

        return new CampaignBrandSurveyWebResponse()
                .withBrandSurveyId(brandSurveyId)
                .withBrandSurveyStatusWeb(
                        new BrandSurveyStatusWeb()
                                .withSurveyStatusDaily(SurveyStatusWeb.valueOf(
                                        brandSurveyStatus.getSurveyStatusDaily().name()))
                                .withReasonIds(brandSurveyStatus.getReasonIds())
                                .withBrandSurveyStopReasonsDaily(mapSet(
                                        brandSurveyStatus.getBrandSurveyStopReasonsDaily(),
                                        brandSurveyStopReason -> BrandSurveyStopReasonWeb.valueOf(brandSurveyStopReason.name())))
                                .withSumSpentByTotalPeriod(brandSurveyStatus.getSumSpentByTotalPeriod())
                                .withSumSpentByDay(brandSurveyStatus.getSumSpentByDay()));
    }
}
