package ru.yandex.direct.web.entity.cashback;

import javax.servlet.http.HttpServletResponse;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.apache.commons.httpclient.HttpStatus;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import ru.yandex.direct.common.util.HttpUtil;
import ru.yandex.direct.core.security.authorization.PreAuthorizeRead;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.result.PathNodeConverterProvider;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;
import ru.yandex.direct.web.entity.cashback.model.CashbackDetailsGroupMode;
import ru.yandex.direct.web.entity.cashback.model.CashbackDetailsInfo;
import ru.yandex.direct.web.entity.cashback.model.CashbackDetailsRequest;
import ru.yandex.direct.web.entity.cashback.model.CashbackDetailsResponse;
import ru.yandex.direct.web.entity.cashback.service.CashbackWebService;
import ru.yandex.direct.web.validation.kernel.ValidationResultConversionService;

import static org.springframework.http.MediaType.APPLICATION_JSON_UTF8_VALUE;
import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;
import static ru.yandex.direct.web.core.security.authentication.DirectCookieAuthProvider.PARAMETER_ULOGIN;
import static ru.yandex.direct.web.entity.cashback.service.CashbackWebService.getMdsFileName;

@Controller
@RequestMapping(value = "/cashback")
@Api(tags = "cashback")
public class CashbackController {
    static final String CASHBACK_DETAILS_CSV_ENDPOINT = "cashback_details_csv";
    static final String CASHBACK_DETAILS_ENDPOINT = "cashback_details";
    static final String PERIOD_MONTHS_PARAM = "period";
    static final String GROUP_BY_PARAM = "groupBy";

    private final DirectWebAuthenticationSource authenticationSource;
    private final CashbackWebService cashbackWebService;

    private final ValidationResultConversionService validationResultConversionService;
    private final PathNodeConverterProvider pathNodeConverterProvider;

    @Autowired
    public CashbackController(DirectWebAuthenticationSource authenticationSource,
                              CashbackWebService cashbackWebService,
                              ValidationResultConversionService validationResultConversionService,
                              PathNodeConverterProvider pathNodeConverterProvider) {
        this.authenticationSource = authenticationSource;
        this.cashbackWebService = cashbackWebService;
        this.validationResultConversionService = validationResultConversionService;
        this.pathNodeConverterProvider = pathNodeConverterProvider;
    }

    @ApiOperation(
            value = "cashbackDetailsCsv",
            httpMethod = "POST",
            nickname = "cashbackDetailsCsv"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params"),
                    @ApiResponse(code = 403, message = "Permission denied"),
            }
    )
    @PreAuthorizeRead
    @PostMapping(value = CASHBACK_DETAILS_CSV_ENDPOINT,
            consumes = APPLICATION_JSON_UTF8_VALUE,
            produces = APPLICATION_JSON_UTF8_VALUE)
    @ResponseBody
    public WebResponse getCashbackDetailsCsv(
            @RequestBody CashbackDetailsRequest request,
            @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin) {
        var auth = authenticationSource.getAuthentication();
        var user = auth.getSubjectUser();

        Result<String> detailsResult = cashbackWebService.getCashbackDetailsCsv(user, request);

        if (hasValidationIssues(detailsResult)) {
            return validationResultConversionService
                    .buildValidationResponse(detailsResult.getValidationResult(), pathNodeConverterProvider);
        }

        return new CashbackDetailsResponse()
                .withResult(new CashbackDetailsInfo().withCsvUrl(detailsResult.getResult()));
    }

    @ApiOperation(
            value = "cashbackDetails",
            httpMethod = "GET",
            nickname = "cashbackDetails"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params"),
                    @ApiResponse(code = 403, message = "Permission denied"),
                    @ApiResponse(code = 404, message = "Not found")
            }
    )
    @PreAuthorizeRead
    @GetMapping(value = CASHBACK_DETAILS_ENDPOINT, produces = "text/csv")
    @ResponseBody
    public void getCashbackDetails(
            @RequestParam(name = PERIOD_MONTHS_PARAM, defaultValue = "6") int periodMonths,
            @RequestParam(name = GROUP_BY_PARAM, required = false) CashbackDetailsGroupMode groupMode,
            @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin,
            HttpServletResponse response) {
        var auth = authenticationSource.getAuthentication();
        var user = auth.getSubjectUser();

        var request = new CashbackDetailsRequest()
                .withPeriod(periodMonths)
                .withGroupBy(groupMode);
        Result<String> detailsResult = cashbackWebService.getCashbackDetailsCsv(user, request);

        if (hasValidationIssues(detailsResult)) {
            response.setStatus(HttpStatus.SC_BAD_REQUEST);
            return;
        }

        if (detailsResult.getResult() == null) {
            response.setStatus(HttpStatus.SC_NOT_FOUND);
        }

        HttpUtil.setInternalRedirectToMdsDirectFiles(response, detailsResult.getResult(), getMdsFileName(user, periodMonths));
    }
}
