package ru.yandex.direct.web.entity.cashback.service;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.univocity.parsers.csv.CsvWriter;
import com.univocity.parsers.csv.CsvWriterSettings;
import one.util.streamex.StreamEx;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.TranslationService;
import ru.yandex.direct.common.mds.MdsHolder;
import ru.yandex.direct.core.entity.cashback.service.CashbackClientsService;
import ru.yandex.direct.core.entity.cashback.service.CashbackProgramsService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.i18n.I18NBundle;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.entity.cashback.CashbackCsvDetailsTranslations;
import ru.yandex.direct.web.entity.cashback.model.CashbackDetailsCsvRow;
import ru.yandex.direct.web.entity.cashback.model.CashbackDetailsGroupMode;
import ru.yandex.direct.web.entity.cashback.model.CashbackDetailsRequest;
import ru.yandex.inside.mds.MdsUrlUtils;
import ru.yandex.misc.io.ByteArrayInputStreamSource;

import static java.nio.charset.StandardCharsets.UTF_8;
import static ru.yandex.direct.core.entity.cashback.CashbackConstants.DETALIZATION_MAX_LENGTH;
import static ru.yandex.direct.core.entity.cashback.CashbackConstants.DETALIZATION_MIN_LENGTH;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.NumberConstraints.inRange;
import static ru.yandex.direct.web.entity.cashback.model.CashbackDetailsGroupMode.BY_MONTH;

@Service
@ParametersAreNonnullByDefault
public class CashbackWebService {
    static final char BOM_HEADER = '\uFEFF';
    private static final DateTimeFormatter MDS_DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyyMMdd-HHmmss");
    private static final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.ofPattern("yyyy-MM");
    private static final Duration EXPIRE_DURATION = Duration.standardHours(6L);

    private static final Logger logger = LoggerFactory.getLogger(CashbackWebService.class);

    private final CashbackProgramsService programsService;
    private final CashbackClientsService clientsService;
    private final TranslationService translationService;

    private final MdsHolder mdsHolder;

    @Autowired
    public CashbackWebService(CashbackProgramsService programsService,
                              CashbackClientsService clientsService,
                              TranslationService translationService,
                              MdsHolder mdsHolder) {
        this.programsService = programsService;
        this.clientsService = clientsService;
        this.translationService = translationService;
        this.mdsHolder = mdsHolder;
    }

    /**
     * Возвращает URL на MDS где лежит CSV детализация начислений кешбэков
     */
    public Result<String> getCashbackDetailsCsv(User user,
                                                CashbackDetailsRequest request) {
        var vr = validateRequest(request);
        if (vr.hasAnyErrors()) {
            return Result.broken(vr);
        }

        var groupMode = request.getGroupBy();
        var details = clientsService.getClientCashbackRewardsDetails(user.getClientId(), request.getPeriod());
        var language = translationService.getLocale().getLanguage();
        var isRuLocale = I18NBundle.RU.getLanguage().equals(language);
        List<CashbackDetailsCsvRow> rows = new ArrayList<>();
        details.getTotalByPrograms().forEach(detailsRow -> {
            if (detailsRow.getProgram() != null) {
                var programName = isRuLocale ? detailsRow.getProgram().getNameRu() :
                        detailsRow.getProgram().getNameEn();
                rows.add(new CashbackDetailsCsvRow()
                        .withProgramId(detailsRow.getProgramId())
                        .withProgramName(programName)
                        .withReward(detailsRow.getRewardWithoutNds())
                        .withDate(detailsRow.getDate()));
            }
        });
        ByteArrayOutputStream outputStream;
        try {
            outputStream = createCsvWrite(rows, groupMode);
        } catch (IOException e) {
            logger.error("Unable to create CSV cashback details", e);
            throw new RuntimeException(e);
        }
        var url = saveToMds(user, request.getPeriod(), outputStream);
        return Result.successful(url);
    }

    private static ValidationResult<CashbackDetailsRequest, Defect> validateRequest(CashbackDetailsRequest request) {
        ItemValidationBuilder<CashbackDetailsRequest, Defect> vb = ItemValidationBuilder.of(request);
        vb.item(request.getPeriod(), CashbackDetailsRequest.PERIOD_FIELD)
                .check(inRange(DETALIZATION_MIN_LENGTH, DETALIZATION_MAX_LENGTH));
        return vb.getResult();
    }

    private ByteArrayOutputStream createCsvWrite(List<CashbackDetailsCsvRow> rows,
                                                 @Nullable CashbackDetailsGroupMode groupMode) throws IOException {
        var outputStream = new ByteArrayOutputStream();
        var outputWriter = new OutputStreamWriter(outputStream, UTF_8);
        outputWriter.write(BOM_HEADER); // Без BOM-заголовка будут проблемы при открытии файла в excel
        var writer = new CsvWriter(outputWriter, new CsvWriterSettings());
        writer.writeHeaders(getCsvHeaders(groupMode));
        writer.writeRowsAndClose(toCsvRows(rows, groupMode).toArray(new String[0][]));
        return outputStream;
    }

    private static List<String[]> toCsvRows(List<CashbackDetailsCsvRow> rows,
                                            @Nullable CashbackDetailsGroupMode groupMode) {
        if (groupMode == null) {
            return mapList(rows, row -> new String[]{
                    row.getProgramId().toString(),
                    row.getProgramName(),
                    row.getReward().toString(),
                    row.getDate().format(DATE_FORMATTER)});
        }

        if (groupMode == BY_MONTH) {
            return getGroupedRows(
                    rows,
                    row -> row.getDate().format(DATE_FORMATTER),
                    CashbackDetailsCsvRow::getProgramName);
        } else { // BY_PROGRAM
            return getGroupedRows(
                    rows,
                    CashbackDetailsCsvRow::getProgramName,
                    row -> row.getDate().format(DATE_FORMATTER));
        }
    }

    private static <T extends String, S extends String> List<String[]> getGroupedRows(
            List<CashbackDetailsCsvRow> rows,
            Function<CashbackDetailsCsvRow, T> groupingBy,
            Function<CashbackDetailsCsvRow, S> secondColumnExtractor) {
        List<String[]> result = new ArrayList<>();
        var groupedRows = StreamEx.of(rows).groupingBy(groupingBy);
        groupedRows.forEach((groupKey, rowsList) -> {
            var isFirstRowInGroup = true;
            for (var row : rowsList) {
                result.add(new String[]{
                        isFirstRowInGroup ? groupKey : "",
                        secondColumnExtractor.apply(row),
                        row.getReward().toString()
                });
                isFirstRowInGroup = false;
            }
        });
        return result;
    }

    private String saveToMds(User user, int periodMonths, ByteArrayOutputStream outputStream) {
        try (TraceProfile profile = Trace.current().profile("cashback_details:save_to_mds")) {
            var data = outputStream.toByteArray();

            var mdsPath = generateMdsPath(user, periodMonths);
            var response = mdsHolder.upload(mdsPath, new ByteArrayInputStreamSource(data), EXPIRE_DURATION);

            return MdsUrlUtils.actionUrl(mdsHolder.getHosts().getHostPortForRead(),
                    "get", mdsHolder.getNamespace().getName(), response.getKey());
        }
    }

    private String[] getCsvHeaders(@Nullable CashbackDetailsGroupMode groupMode) {
        if (groupMode == null) {
            return new String[]{
                    translationService.translate(CashbackCsvDetailsTranslations.INSTANCE.programId()),
                    translationService.translate(CashbackCsvDetailsTranslations.INSTANCE.programName()),
                    translationService.translate(CashbackCsvDetailsTranslations.INSTANCE.reward()),
                    translationService.translate(CashbackCsvDetailsTranslations.INSTANCE.date())
            };
        }
        if (groupMode == BY_MONTH) {
            return new String[]{
                    translationService.translate(CashbackCsvDetailsTranslations.INSTANCE.date()),
                    translationService.translate(CashbackCsvDetailsTranslations.INSTANCE.programName()),
                    translationService.translate(CashbackCsvDetailsTranslations.INSTANCE.reward()),
            };
        } else { // BY_PROGRAM
            return new String[]{
                    translationService.translate(CashbackCsvDetailsTranslations.INSTANCE.programName()),
                    translationService.translate(CashbackCsvDetailsTranslations.INSTANCE.date()),
                    translationService.translate(CashbackCsvDetailsTranslations.INSTANCE.reward()),
            };
        }
    }

    private static String generateMdsPath(User user, int periodMonths) {
        var time = MDS_DATE_TIME_FORMATTER.format(LocalDateTime.now());
        return String.format("cashback_details/%d/%s/%s",
                user.getClientId().asLong(), time, getMdsFileName(user, periodMonths));
    }

    public static String getMdsFileName(User user, int periodMonths) {
        return String.format("cashback_details_last_%d_months_%s.csv", periodMonths, user.getLogin());
    }
}
