package ru.yandex.direct.web.entity.communication.controller;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseCookie;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.CookieValue;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import ru.yandex.ads.bsyeti.libs.communications.EMessageStatus;
import ru.yandex.direct.common.enums.YandexDomain;
import ru.yandex.direct.common.util.HostUtils;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.web.annotations.AllowedOperatorRoles;
import ru.yandex.direct.web.annotations.AllowedSubjectRoles;
import ru.yandex.direct.web.core.model.WebErrorResponse;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.model.WebSuccessResponse;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;
import ru.yandex.direct.web.entity.communication.model.EventMessageStatusesHolder;
import ru.yandex.direct.web.entity.communication.model.SendEventRequest;
import ru.yandex.direct.web.entity.communication.model.response.WebGenerateMessageResult;
import ru.yandex.direct.web.entity.communication.service.CommunicationWebService;

import static org.springframework.http.MediaType.APPLICATION_JSON_UTF8_VALUE;
import static ru.yandex.direct.rbac.RbacRole.AGENCY;
import static ru.yandex.direct.rbac.RbacRole.CLIENT;
import static ru.yandex.direct.rbac.RbacRole.INTERNAL_AD_ADMIN;
import static ru.yandex.direct.rbac.RbacRole.INTERNAL_AD_MANAGER;
import static ru.yandex.direct.rbac.RbacRole.INTERNAL_AD_SUPERREADER;
import static ru.yandex.direct.rbac.RbacRole.MANAGER;
import static ru.yandex.direct.rbac.RbacRole.MEDIA;
import static ru.yandex.direct.rbac.RbacRole.PLACER;
import static ru.yandex.direct.rbac.RbacRole.SUPER;
import static ru.yandex.direct.rbac.RbacRole.SUPERREADER;
import static ru.yandex.direct.rbac.RbacRole.SUPPORT;
import static ru.yandex.direct.utils.JsonUtils.fromJson;

/*
 * Контроллер для работы с коммуникационной платформой
 */
@Api(tags = "communication")
@Controller
@ParametersAreNonnullByDefault
@RequestMapping("/communication")
@AllowedSubjectRoles({SUPER, SUPERREADER, SUPPORT, PLACER, MEDIA, MANAGER, AGENCY,
        INTERNAL_AD_ADMIN, INTERNAL_AD_MANAGER, INTERNAL_AD_SUPERREADER, CLIENT})
public class CommunicationController {
    private final CommunicationWebService communicationWebService;
    private final DirectWebAuthenticationSource authenticationSource;
    private final FeatureService featureService;

    private static final Logger logger = LoggerFactory.getLogger(CommunicationController.class);

    private static final String COOKIE_PATH = "/web-api/communication/messages";

    private static final String ADDED_STATUSES_COOKIE_NAME = "added-message-statuses";

    private static final String REMOVED_STATUSES_COOKIE_NAME = "removed-message-statuses";

    private static final Integer COOKIE_TTL_SECONDS = 60;

    private static final YandexDomain DEFAULT_TLD_DOMAIN = YandexDomain.RU;


    @Autowired
    public CommunicationController(CommunicationWebService communicationWebService,
                                   DirectWebAuthenticationSource authenticationSource,
                                   FeatureService featureService) {
        this.communicationWebService = communicationWebService;
        this.authenticationSource = authenticationSource;
        this.featureService = featureService;
    }

    @ApiOperation(
            value = "Send new communication event",
            httpMethod = "POST",
            nickname = "event"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = WebSuccessResponse.class)
            }
    )
    @RequestMapping(
            path = "/event",
            method = RequestMethod.POST
    )
    @ResponseBody
    @AllowedOperatorRoles({SUPER})
    public WebResponse sendEvent(
            @RequestBody String rawRequest) {
        SendEventRequest request = fromJson(rawRequest, SendEventRequest.class);
        return communicationWebService.sendEvent(request);
    }

    @ApiOperation(
            value = "Send new status for message",
            httpMethod = "POST",
            nickname = "event/status"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = WebSuccessResponse.class)
            }
    )
    @RequestMapping(
            path = "/event/status",
            method = RequestMethod.POST
    )
    @ResponseBody
    public WebResponse sendEventChange(
            @ApiParam(value = "Id ообщения")
            @RequestParam(value = "MessageId") Long messageId,
            @ApiParam(value = "Новый статус для изменения")
            @RequestParam(value = "Status") String status) {

        User operator = authenticationSource.getAuthentication().getOperator();
        return communicationWebService.changeMessageStatus(messageId, status, operator.getUid());
    }


    @ApiOperation(
            value = "Get messages by User",
            httpMethod = "GET",
            nickname = "messages"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = WebSuccessResponse.class)
            }
    )
    @RequestMapping(
            path = "/messages",
            method = RequestMethod.GET,
            produces = APPLICATION_JSON_UTF8_VALUE
    )
    @ResponseBody
    public WebResponse getMessages(
            @CookieValue(name = ADDED_STATUSES_COOKIE_NAME, required = false) String addedStatusesCookie,
            @CookieValue(name = REMOVED_STATUSES_COOKIE_NAME, required = false) String removedStatusesCookie) {
        User operator = authenticationSource.getAuthentication().getOperator();
        User subjectUser = authenticationSource.getAuthentication().getSubjectUser();
        if (!featureService.isEnabled(operator.getUid(), FeatureName.COMMUNICATION_MESSAGES_ENABLED)) {
           return new WebGenerateMessageResult(Collections.emptyList());
        }
        return communicationWebService.getUserMessages(
                subjectUser.getClientId(),
                operator.getUid(),
                parseCookieValue(removedStatusesCookie),
                parseCookieValue(addedStatusesCookie));
    }


    @ApiOperation(
            value = "Send success",
            httpMethod = "POST",
            nickname = "messages"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = WebSuccessResponse.class)
            }
    )
    @RequestMapping(
            path = "/messages/{messageId}/button",
            method = RequestMethod.POST,
            produces = APPLICATION_JSON_UTF8_VALUE
    )
    @ResponseBody
    public ResponseEntity<WebResponse> handleMessageIdAndButtonId(
            HttpServletRequest request, HttpServletResponse response,
            @CookieValue(name = ADDED_STATUSES_COOKIE_NAME, required = false) String addedStatuses,
            @CookieValue(name = REMOVED_STATUSES_COOKIE_NAME, required = false) String removedStatuses,
            @NotNull @PathVariable("messageId") Long messageId,
            @RequestParam(name = "id", required = false) Integer buttonId) {
        var auth = authenticationSource.getAuthentication();
        User operator = auth.getOperator();
        var subjectUser = auth.getSubjectUser();
        String hostName = request.getServerName();
        EventMessageStatusesHolder result = communicationWebService.handleActionButton(
                buttonId, messageId, operator, subjectUser.getClientId(),
                parseCookieValue(addedStatuses),
                parseCookieValue(removedStatuses));
        if (result != null) {
            var addedStatusesCookie = buildCookie(hostName, ADDED_STATUSES_COOKIE_NAME, addedStatuses,
                    messageId, result.getAddedStatuses());
            var removedStatusesCookie = buildCookie(hostName, REMOVED_STATUSES_COOKIE_NAME,
                    removedStatuses, messageId, result.getRemovedStatuses());

            return ResponseEntity.ok().header(HttpHeaders.SET_COOKIE, addedStatusesCookie, removedStatusesCookie)
                    .body(new WebSuccessResponse());
        } else {
            return new ResponseEntity<>(new WebErrorResponse(500, "Request cannot be processed"),
                    HttpStatus.OK);
        }
    }

    private String buildCookie(@Nullable String hostName, @NotNull String cookieName,
                               @Nullable String existingCookieValue, @NotNull Long messageId,
                               @Nullable Set<EMessageStatus> statuses){
        if (statuses == null){
            return null;
        }

        return ResponseCookie.from(cookieName, buildCookieValue(existingCookieValue, messageId, statuses))
                .maxAge(COOKIE_TTL_SECONDS)
                .secure(true)
                .httpOnly(true)
                .path(COOKIE_PATH)
                .domain(getTldDomain(hostName))
                .build()
                .toString();
    }

    private String getTldDomain(@Nullable String hostName){
        return HostUtils.getYandexDomain(hostName).orElse(DEFAULT_TLD_DOMAIN).getYandexDomain();
    }

    String buildCookieValue(@Nullable String cookieValue,
                            @NotNull Long messageId,
                            @NotNull Set<EMessageStatus> statuses){
        var messages = parseCookieValue(cookieValue);
        var messageStatuses = messages.getOrDefault(messageId, new HashSet<>());
        messageStatuses.addAll(statuses);
        messages.put(messageId, messageStatuses);

        return buildCookieValue(messages);
    }

    /**
     * Собираем значение куки на основе идентификаторов сообщений и их статусов. Формат куки:
     * messageId1=status1+status2+status3|messageId2=status1|messageId3=status3
     * В качестве разделителей используются символы допустимые с точки зрения RFC 2109, 2965, 6265
     * @param messagesStatuses map где в качестве ключа используется messageId, а в качестве значения - набор статусов
     * @return значение куки
     */
    private String buildCookieValue(@Nullable Map<Long, Set<EMessageStatus>> messagesStatuses){
        if (messagesStatuses == null || messagesStatuses.isEmpty()){
            return null;
        }

        return messagesStatuses.entrySet().stream()
                .filter(e -> Objects.nonNull(e.getValue()) && !e.getValue().isEmpty())
                .sorted(Map.Entry.comparingByKey())
                .map(e -> e.getKey() + "-" + mapStatusesToPlainString(e.getValue()))
                .collect(Collectors.joining("|"));
    }

    /**
     * Парсинг куки в которой сохранены обработанные статусы в формате:
     * messageId1=status1+status2+status3|messageId2=status1|messageId3=status3
     * @param cookieValue значение куки
     * @return map где в качестве ключа используется messageId, а в качестве значения - набор статусов
     */
    private Map<Long, Set<EMessageStatus>> parseCookieValue(@Nullable String cookieValue){
        if (cookieValue == null){
            return new HashMap<>();
        }

        return Stream.of(cookieValue.split("\\|"))
                .map(String::trim)
                .filter(Predicate.not(String::isEmpty))
                .map(this::parseMessagesFromString)
                .filter(Objects::nonNull)
                .collect(Collectors.toMap(Pair::getLeft, Pair::getRight, (v1, v2) -> v2));
    }

    private Pair<Long, Set<EMessageStatus>> parseMessagesFromString(@NotNull String messageKeyValue){
        var keyValue = messageKeyValue.split("-");
        if (keyValue.length != 2 || keyValue[0].isBlank() || keyValue[1].isBlank()){
            return null;
        }

        try {
            Long messageId = Long.valueOf(keyValue[0].trim());

            var statuses = Stream.of(keyValue[1].split("\\+")).map(String::trim)
                    .map(EMessageStatus::valueOf).collect(Collectors.toSet());

            return Pair.of(messageId, statuses);
        } catch (RuntimeException ex){
            logger.error("Не удалось распарсить cookie", ex);
            return null;
        }
    }

    private String mapStatusesToPlainString(@Nullable Set<EMessageStatus> statuses){
        if (statuses == null || statuses.isEmpty()){
            return "";
        }

        return statuses.stream().map(EMessageStatus::name).sorted().collect(Collectors.joining("+"));
    }
}
