package ru.yandex.direct.web.entity.communication.service;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.InvalidProtocolBufferException;
import one.util.streamex.StreamEx;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.ads.bsyeti.libs.communications.EChannel;
import ru.yandex.ads.bsyeti.libs.communications.EEventType;
import ru.yandex.ads.bsyeti.libs.communications.EMessageStatus;
import ru.yandex.ads.bsyeti.libs.communications.ESourceType;
import ru.yandex.ads.bsyeti.libs.communications.EWebUIButtonStyle;
import ru.yandex.ads.bsyeti.libs.communications.TEventSource;
import ru.yandex.ads.bsyeti.libs.events.TCommunicationEvent;
import ru.yandex.ads.bsyeti.libs.events.TCommunicationEventData;
import ru.yandex.direct.communication.CommunicationChannelRepository;
import ru.yandex.direct.communication.CommunicationClient;
import ru.yandex.direct.communication.container.CommunicationChannelItem;
import ru.yandex.direct.communication.container.web.Body;
import ru.yandex.direct.communication.container.web.Button;
import ru.yandex.direct.communication.container.web.CommunicationMessage;
import ru.yandex.direct.communication.container.web.CommunicationMessageButton;
import ru.yandex.direct.communication.container.web.Footer;
import ru.yandex.direct.communication.container.web.Image;
import ru.yandex.direct.communication.container.web.Message;
import ru.yandex.direct.communication.container.web.Slide;
import ru.yandex.direct.communication.container.web.WebMessage;
import ru.yandex.direct.communication.service.CommunicationChannelService;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersion;
import ru.yandex.direct.core.entity.communication.model.CommunicationEventVersionStatus;
import ru.yandex.direct.core.entity.communication.repository.CommunicationEventVersionsRepository;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.web.core.model.WebErrorResponse;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.model.WebSuccessResponse;
import ru.yandex.direct.web.entity.communication.model.EventMessageStatusesHolder;
import ru.yandex.direct.web.entity.communication.model.SendEventRequest;
import ru.yandex.direct.web.entity.communication.model.response.WebGenerateMessageResult;

import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;


/**
 * Сервис для работы с коммуникационной платформой
 */
@Service
@ParametersAreNonnullByDefault
public class CommunicationWebService {
    private static final int DATA_ID = 1;
    private static final Logger logger = LoggerFactory.getLogger(
            CommunicationWebService.class);

    private final CommunicationClient communicationClient;
    private final CommunicationEventVersionsRepository versionsRepository;
    private final CommunicationChannelService communicationChannelService;
    private final CommunicationChannelRepository communicationChannelRepository;

    @Autowired
    public CommunicationWebService(
            CommunicationClient communicationClient,
            CommunicationEventVersionsRepository communicationEventVersionsRepository,
            CommunicationChannelService communicationChannelService,
            CommunicationChannelRepository communicationChannelRepository) {
        this.communicationClient = communicationClient;
        versionsRepository = communicationEventVersionsRepository;
        this.communicationChannelService = communicationChannelService;
        this.communicationChannelRepository = communicationChannelRepository;
    }

    public WebResponse changeMessageStatus(Long messageId, String status, Long uid) {
        EMessageStatus statusEmun;
        try {
            statusEmun = EMessageStatus.valueOf(status);
        } catch (IllegalArgumentException e) {
            logger.error(String.format("Невалидные статус %s", status), e);
            return new WebErrorResponse(400, "Couldn't cast status: " + status);
        }

        TCommunicationEvent event = TCommunicationEvent.newBuilder()
                .addUids(uid)
                .setTimestamp(System.currentTimeMillis())
                .setData(TCommunicationEventData.newBuilder()
                        .setId(DATA_ID)
                        .setType(EEventType.MODIFY_MESSAGES)
                        .setSource(TEventSource.newBuilder()
                                .setId(uid)
                                .setType(ESourceType.DIRECT_WEB)
                                .build())
                        .setModifyMessages(TCommunicationEventData.TModifyMessages.newBuilder()
                                .addTargetChannels(EChannel.DIRECT_WEB_UI)
                                .setMessageId(messageId)
                                .addStatus(statusEmun)
                                .build())
                        .build())
                .build();

        try {
            if (!communicationClient.send(event)) {
                return new WebErrorResponse(500, "Problem with event numeration.");
            }
        } catch (Exception e) {
            logger.error("Не удалось отправить событие", e);
            return new WebErrorResponse(500, "Couldn't send event.");
        }
        return new WebSuccessResponse();
    }

    public WebResponse sendEvent(SendEventRequest request) {
        TCommunicationEvent event;
        try {
            event = request.getParsedEvent();
        } catch (InvalidProtocolBufferException e) {
            logger.error("Не удалось распарсить событие", e);
            return new WebErrorResponse(400, "Invalid event structure");
        }
        try {
            if (!communicationClient.send(event)) {
                return new WebErrorResponse(500, "Problem with event numeration.");
            }
        } catch (Exception e) {
            logger.error("Не удалось отправить событие", e);
            return new WebErrorResponse(500, "Couldn't send event: " + e.getMessage());
        }

        return new WebSuccessResponse();
    }

    public EventMessageStatusesHolder handleActionButton(
            @Nullable Integer buttonId,
            Long messageId,
            User operator,
            ClientId clientId,
            @NotNull Map<Long, Set<EMessageStatus>> addedStatuses,
            @NotNull Map<Long, Set<EMessageStatus>> removedStatuses
    ) {
        logger.info("messageID= {} buttonId= {}", messageId, buttonId);

        var additionalInfo = communicationChannelService
                .getAdditionalInfoContainer(clientId, operator.getUid(), 0L, "ru")
                .withCookieRemovedStatusesByMessageId((Map) removedStatuses)
                .withCookieAddedStatusesByMessageId((Map) addedStatuses);
        var res = communicationChannelService
                .getCommunicationMessageAndHandleButton(additionalInfo, "popup", messageId, buttonId);
        if (res != null) {
            return new EventMessageStatusesHolder().withAddedStatuses(res);
        }

        Long uid = operator.getUid();
        var eventIdWithStatuses = resolveNewMessageStatus(buttonId, messageId, uid);
        if (eventIdWithStatuses == null) {
            logger.error("messageID {} was not found", messageId);
            return null;
        }
        TCommunicationEvent event = TCommunicationEvent.newBuilder()
                .addUids(uid)
                .setTimestamp(System.currentTimeMillis())
                .setData(TCommunicationEventData.newBuilder()
                        .setId(DATA_ID)
                        .setType(EEventType.MODIFY_MESSAGES)
                        .setSource(TEventSource.newBuilder()
                                .setId(uid)
                                .setType(ESourceType.DIRECT_WEB)
                                .build())
                        .setModifyMessages(TCommunicationEventData.TModifyMessages.newBuilder()
                                .addTargetChannels(EChannel.DIRECT_WEB_UI)
                                .setTargetEventId(eventIdWithStatuses.getEventId())
                                .addAllStatus(eventIdWithStatuses.getActualStatuses())
                                .build())
                        .build())
                .build();

        try {
            communicationClient.send(event);
        } catch (Exception e) {
            logger.error("Не удалось отправить событие", e);
            return null;
        }
        return eventIdWithStatuses;
    }

    @Nullable
    private EventMessageStatusesHolder resolveNewMessageStatus(@Nullable Integer buttonId, Long messageId,
                                                               Long uid) {
        var channelItem = communicationChannelRepository
                .getMessagesByIds(List.of(messageId)).stream().filter(i -> uid.equals(i.getUid())).findAny();
        if (channelItem.isEmpty()) {
            return null;
        }
        var statuses = channelItem.map(CommunicationChannelItem::getStatuses);
        var eventId = channelItem.map(CommunicationChannelItem::getEventId);
        if (statuses.isEmpty()) {
            return null;
        }
        Set<EMessageStatus> resultStatuses = new HashSet<>(statuses.get());
        Set<EMessageStatus> addedStatuses = new HashSet<>();
        Set<EMessageStatus> removedStatuses = new HashSet<>();

        resultStatuses.add(EMessageStatus.DELIVERED);
        addedStatuses.add(EMessageStatus.DELIVERED);

        List<Slide> slides = channelItem
                .map(CommunicationChannelItem::getWebMessage)
                .map(WebMessage::getMessage)
                .map(Message::getSlides)
                .orElse(Collections.emptyList());
        boolean isInfoLinkButton = buttonId != null && StreamEx.of(slides)
                .map(Slide::getFooter)
                .filter(Objects::nonNull)
                .map(Footer::getButtons)
                .flatMap(Collection::stream)
                .anyMatch(b -> b.getId() == buttonId && "infoLink".equals(b.getAction()));
        if (!isInfoLinkButton) {
            resultStatuses.remove(EMessageStatus.NEW);
            removedStatuses.add(EMessageStatus.NEW);
        }

        return new EventMessageStatusesHolder().withEventId(eventId.get().intValue()).withActualStatuses(resultStatuses)
                        .withAddedStatuses(addedStatuses).withRemovedStatuses(removedStatuses);
    }
    public WebSuccessResponse getUserMessages(
            Long uid,
            @NotNull Map<Long, Set<EMessageStatus>> removedStatuses
    ) {
        return getUserMessages(null, uid, removedStatuses, Map.of());
    }

    /**
     * Получить список сообщений
     * @param uid идентификатор пользователя
     * @param removedStatuses список из куки о ранее убранных статусах
     * @param removedStatuses список из куки о ранее добавленных статусах
     */
    public WebSuccessResponse getUserMessages(
            ClientId clientId,
            Long uid,
            @NotNull Map<Long, Set<EMessageStatus>> removedStatuses,
            @NotNull Map<Long, Set<EMessageStatus>> addedStatuses) {
        int currentTimeSeconds = (int) (System.currentTimeMillis() / 1000);
        var activeEventIds = versionsRepository
                .getVersionsByStatuses(List.of(CommunicationEventVersionStatus.ACTIVE))
                .stream()
                .map(CommunicationEventVersion::getEventId)
                .distinct()
                .collect(Collectors.toList());
        List<CommunicationChannelItem> messagesRowList = communicationChannelRepository.getMessagesByUid(uid, activeEventIds);

        var messages = StreamEx.of(messagesRowList)
                .filter(message -> message.getExpired() > currentTimeSeconds)
                .filter(message -> message.getChannel() == EChannel.DIRECT_WEB_UI.getNumber())
                .filter(message -> message.getStatuses() != null)
                .filter(message -> message.getStatuses().contains(EMessageStatus.NEW))
                //проверяем по куке что статус NEW еще не был удален
                .filter(message -> removedStatuses.get(message.getMessageId()) == null ||
                        !removedStatuses.get(message.getMessageId()).contains(EMessageStatus.NEW))
                .map(CommunicationChannelItem::getWebMessage)
                .nonNull()
                .filter(message -> message.getShowAfter() <= currentTimeSeconds / 60) // отфильтровать значения
                // showAfter меньше времени в минутах
                .sortedByInt(WebMessage::getPriority)
                .map(WebMessage::getMessage)
                .nonNull()
                .toList();

        if (clientId == null) {
            return new WebGenerateMessageResult(messages);
        }

        var additionalInfo = communicationChannelService
                .getAdditionalInfoContainer(clientId, uid, null, "ru")
                .withCookieRemovedStatusesByMessageId((Map) removedStatuses)
                .withCookieAddedStatusesByMessageId((Map) addedStatuses);
        messages = StreamEx.of(communicationChannelService
                .getCommunicationMessage(additionalInfo, Set.of("popup"), List.of(), true))
                .map(CommunicationWebService::toPopupMessage)
                .append(messages)
                .toList();

        return new WebGenerateMessageResult(messages);
    }

    private static Message toPopupMessage(CommunicationMessage message) {
        var content = message.getContent();
        var imageUrl = content.getImageUrl();
        var title = content.getTitle();
        var text = content.getText();
        var image = imageUrl == null ? null : new Image()
                .withUrl(imageUrl);
        var body = title == null && text == null ? null : new Body()
                .withTitle(title)
                .withText(text);
        var buttons = filterAndMapList(
                content.getButtons(),
                b -> !EWebUIButtonStyle.BUTTON_STYLE_CLOSE.equals(b.getStyle()),
                CommunicationWebService::toPopupButton);
        var footer = new Footer()
                .withButtons(buttons);
        var slide = new Slide()
                .withImage(image)
                .withBody(body)
                .withFooter(footer);
        return new Message()
                .withId(message.getMessageId().toString())
                .withSlides(List.of(slide));
    }

    private static Button toPopupButton(CommunicationMessageButton button) {
        return new Button()
                .withId((int) button.getId())
                .withText(button.getTitle())
                .withHref(button.getHref())
                .withAction("actionLink")
                .withView("action")
                .withInLine(false);
    }
}
