package ru.yandex.direct.web.entity.contentpromotion.common.controller;

import java.util.UUID;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import ru.yandex.direct.core.entity.contentpromotion.ContentPromotionMetaWithStatus;
import ru.yandex.direct.core.entity.contentpromotion.ContentPromotionService;
import ru.yandex.direct.core.entity.contentpromotion.ContentStatus;
import ru.yandex.direct.core.entity.contentpromotion.model.ContentPromotionContentType;
import ru.yandex.direct.core.security.authorization.PreAuthorizeRead;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.core.model.WebErrorResponse;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.security.captcha.DisableAutoCaptcha;
import ru.yandex.direct.web.entity.contentpromotion.common.model.ContentPromotionTypeForGetMeta;
import ru.yandex.direct.web.entity.contentpromotion.common.model.GetContentPromotionMetaRequest;
import ru.yandex.direct.web.entity.contentpromotion.common.model.GetContentPromotionMetaResponse;
import ru.yandex.direct.web.entity.contentpromotion.common.service.ContentPromotionValidationService;
import ru.yandex.direct.web.validation.kernel.ValidationResultConversionService;

import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.web.core.security.authentication.DirectCookieAuthProvider.PARAMETER_ULOGIN;
import static ru.yandex.direct.web.entity.contentpromotion.common.model.ContentPromotionTypeForGetMeta.VIDEO;

@Controller
@RequestMapping(value = "/content/promotion",
        produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Api(tags = "content-promotion")
public class ContentPromotionController {

    private final ContentPromotionService service;
    private final ContentPromotionValidationService validationService;
    private final ValidationResultConversionService validationResultConversionService;

    public ContentPromotionController(ContentPromotionService service,
                                      ContentPromotionValidationService validationService,
                                      ValidationResultConversionService validationResultConversionService) {
        this.service = service;
        this.validationService = validationService;
        this.validationResultConversionService = validationResultConversionService;
    }

    @ApiOperation(
            value = "getContentPromotionMeta",
            httpMethod = "POST",
            nickname = "getContentPromotionMeta")
    @ApiResponses(
            {
                    @ApiResponse(code = 200, message = "Ok", response = GetContentPromotionMetaResponse.class),
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 501, message = "Not implemented", response = WebErrorResponse.class),
                    @ApiResponse(code = 504, message = "Gateway Timeout", response = WebErrorResponse.class),
            })
    @PreAuthorizeRead
    @DisableAutoCaptcha
    @PostMapping(path = "/getContentPromotionMeta")
    @ResponseBody
    public ResponseEntity<WebResponse> getContentPromotionMeta(
            @RequestBody GetContentPromotionMetaRequest request,
            @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin) {
        String requestId = UUID.randomUUID().toString().toUpperCase();

        ValidationResult<GetContentPromotionMetaRequest, Defect> vr = validationService.validateGetMetaRequest(request);
        if (vr.hasAnyErrors()) {
            return new ResponseEntity<>(validationResultConversionService.buildValidationResponse(vr),
                    HttpStatus.BAD_REQUEST);
        }

        ContentPromotionContentType coreContentType = ContentPromotionTypeForGetMeta.toCoreValue(
                nvl(request.getContentPromotionTypeForGetMeta(), VIDEO));
        ContentPromotionMetaWithStatus metaWithStatus = service.getMeta(request.getUrl(),
                requestId,
                coreContentType,
                request.getCampaignId(),
                request.getAdGroupId(),
                subjectLogin);

        if (metaWithStatus.getContentStatus() == ContentStatus.NOT_FOUND) {
            return new ResponseEntity<>(metaWithStatus.getContentStatus().getStatus());
        }
        GetContentPromotionMetaResponse response = new GetContentPromotionMetaResponse(requestId,
                metaWithStatus.getMeta(),
                validationResultConversionService.buildWebValidationResult(metaWithStatus.getValidationResult()));
        return new ResponseEntity<>(response, metaWithStatus.getContentStatus().getStatus());
    }
}
