package ru.yandex.direct.web.entity.cookie.banneraiming.controller;

import java.io.IOException;
import java.time.Duration;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;

import ru.yandex.direct.common.util.HttpUtil;
import ru.yandex.direct.web.annotations.OperatorHasFeatures;
import ru.yandex.direct.web.core.security.csrf.CsrfCheck;
import ru.yandex.direct.web.entity.cookie.banneraiming.service.BannerAimingService;

import static ru.yandex.direct.feature.FeatureName.BANNER_AIMING_ALLOWED;
import static ru.yandex.direct.feature.FeatureName.BANNER_AIMING_CPM_ALLOWED;
import static ru.yandex.direct.feature.FeatureName.BANNER_AIMING_CPM_YNDX_FRONTPAGE_ALLOWED;
import static ru.yandex.direct.web.core.security.authentication.DirectCookieAuthProvider.PARAMETER_ULOGIN;

@Controller
@RequestMapping(value = "/banner_aiming")
@Api("Api to work with adoc cookie, which is for banner aiming at yourself")
public class BannerAimingController {
    private static final String PARAMETER_BID = "bid";
    private static final String PARAMETER_RETPATH = "retpath";
    private static final String PARAMETER_COOKIE_TTL = "cookie_ttl";

    private final BannerAimingService bannerAimingService;

    @Autowired
    public BannerAimingController(BannerAimingService bannerAimingService) {
        this.bannerAimingService = bannerAimingService;
    }

    @ApiOperation(
            value = "Set adoc cookie",
            httpMethod = "GET",
            nickname = "setCookie"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params"),
                    @ApiResponse(code = 403, message = "Permission denied"),
                    @ApiResponse(code = 302, message = "Ok, redirect for set cookie")
            }
    )

    @GetMapping(
            path = "/set_cookie"
    )
    @CsrfCheck(enabled = true)
    @OperatorHasFeatures({BANNER_AIMING_ALLOWED})
    @OperatorHasFeatures({BANNER_AIMING_CPM_ALLOWED})
    @OperatorHasFeatures({BANNER_AIMING_CPM_YNDX_FRONTPAGE_ALLOWED})
    public void setCookie(HttpServletRequest request, HttpServletResponse response,
                          @SuppressWarnings("unused")

                          @ApiParam(value = "URL to redirect after cookie setting", required = true)
                          @RequestParam(value = PARAMETER_RETPATH) String retPath,

                          @ApiParam(value = "Banner id for aiming", required = true)
                          @RequestParam(value = PARAMETER_BID) long bid,

                          @ApiParam(value = "Cookie ttl in minutes", required = true)
                          @RequestParam(value = PARAMETER_COOKIE_TTL) int cookieTtlminutes,

                          @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin) {
        String yandexUid = HttpUtil.getCookieValue("yandexuid", request).orElse(null);
        String host = request.getServerName();
        Duration cookieTtl = Duration.ofMinutes(cookieTtlminutes);
        String redirectURL = bannerAimingService.getSetCookieUrl(host, yandexUid, bid, retPath, cookieTtl);
        try {
            response.sendRedirect(redirectURL);
        } catch (IOException e) {
            throw new IllegalStateException(e);

        }
    }

    @ApiOperation(
            value = "Reset adoc cookie",
            httpMethod = "GET",
            nickname = "resetCookie"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params"),
                    @ApiResponse(code = 302, message = "Ok, redirect for reset cookie")
            }
    )

    @GetMapping(
            path = "/reset_cookie"
    )
    @CsrfCheck(enabled = true)
    @OperatorHasFeatures({BANNER_AIMING_ALLOWED})
    @OperatorHasFeatures({BANNER_AIMING_CPM_ALLOWED})
    @OperatorHasFeatures({BANNER_AIMING_CPM_YNDX_FRONTPAGE_ALLOWED})
    public void resetCookie(HttpServletRequest request, HttpServletResponse response,
                            @SuppressWarnings("unused")

                            @ApiParam(value = "URL to redirect after cookie setting", required = true)
                            @RequestParam(value = PARAMETER_RETPATH) String retPath,
                            @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin) {
        String host = request.getServerName();
        String redirectURL = bannerAimingService.getResetCookieUrl(host, retPath);
        try {
            response.sendRedirect(redirectURL);
        } catch (IOException e) {
            throw new IllegalStateException(e);

        }
    }
}
