package ru.yandex.direct.web.entity.cookie.banneraiming.service;

import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.Base64;
import java.util.Optional;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;

import org.apache.commons.codec.binary.Hex;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.enums.YandexDomain;
import ru.yandex.direct.common.util.HostUtils;
import ru.yandex.direct.common.util.TuneSecretKey;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerRelationsRepository;
import ru.yandex.direct.core.security.AccessDeniedException;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.liveresource.LiveResource;
import ru.yandex.direct.liveresource.provider.LiveResourceFactoryBean;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.utils.net.FastUrlBuilder;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;
import ru.yandex.direct.web.entity.cookie.banneraiming.container.SecretKey;

import static com.google.common.base.Preconditions.checkState;

@Service
public class BannerAimingService {

    private final Clock clock;
    private final DirectWebAuthenticationSource authenticationSource;
    private final LiveResource secretKeyResource;
    private final BannerCommonRepository bannerCommonRepository;
    private final BannerRelationsRepository bannerRelationsRepository;
    private final ShardHelper shardHelper;
    private static final String COOKIE_NAME = "adoc";
    private static final String DEFAULT_YANDEX_DOMAIN = "yandex.ru";
    private static final String EMPTY_VALUE = "";
    private static final Duration MAX_COOKIE_TTL = Duration.ofMinutes(30);

    @Autowired
    public BannerAimingService(DirectWebAuthenticationSource authenticationSource,
                               @Value("${cookie.banner_aiming_key_path}") String signKeyResource,
                               BannerCommonRepository bannerCommonRepository,
                               BannerRelationsRepository bannerRelationsRepository,
                               ShardHelper shardHelper,
                               LiveResourceFactoryBean liveResourceFactoryBean) {
        this(authenticationSource, signKeyResource, bannerCommonRepository,
                bannerRelationsRepository, shardHelper,
                Clock.systemUTC(), liveResourceFactoryBean);
    }

    public BannerAimingService(DirectWebAuthenticationSource authenticationSource,
                               @Value("${cookie.banner_aiming_key_path}") String signKeyResource,
                               BannerCommonRepository bannerCommonRepository,
                               BannerRelationsRepository bannerRelationsRepository,
                               ShardHelper shardHelper,
                               Clock clock,
                               LiveResourceFactoryBean liveResourceFactoryBean) {
        this.authenticationSource = authenticationSource;
        this.secretKeyResource = liveResourceFactoryBean.get(signKeyResource);
        this.bannerCommonRepository = bannerCommonRepository;
        this.bannerRelationsRepository = bannerRelationsRepository;
        this.shardHelper = shardHelper;
        this.clock = clock;
    }

    private boolean isBannerBelongToClient(long bid) {
        ClientId clientId = authenticationSource.getAuthentication().getSubjectUser().getClientId();
        int shard = shardHelper.getShardByClientId(clientId);
        return bannerRelationsRepository.bannerBelongsToClient(shard, bid, clientId);
    }

    private Long getBannerIdForAiming(long bid) {
        ClientId clientId = authenticationSource.getAuthentication().getSubjectUser().getClientId();
        int shard = shardHelper.getShardByClientId(clientId);
        return bannerCommonRepository.getBsBannerIdForAiming(shard, bid);
    }

    String getSecretKeyForSign(long timestamp) {
        String secretKeysJson = secretKeyResource.getContent();

        SecretKey[] secretKeys = JsonUtils.fromJson(secretKeysJson, SecretKey[].class);
        SecretKey secretKey = Arrays.stream(secretKeys)
                .filter(sk -> timestamp >= sk.getTimestampFrom() && timestamp < sk.getTimestampTo())
                .findAny()
                .orElseThrow(() -> new IllegalStateException("No one correct secret key"));
        return secretKey.getData();
    }

    String calculateSign(long expire, long timestamp, String yandexUid, long bannerId) {
        String data = expire + ":" + timestamp + ":" + yandexUid + ":" + bannerId;
        String sign;
        try {
            Mac sha1HMAC = Mac.getInstance("HmacSHA1");
            String keyForSignString = getSecretKeyForSign(timestamp);
            byte[] keyForSignBase64 = Base64.getDecoder().decode(keyForSignString);
            SecretKeySpec secretKey = new SecretKeySpec(keyForSignBase64, sha1HMAC.getAlgorithm());
            sha1HMAC.init(secretKey);
            sign = Hex.encodeHexString(sha1HMAC.doFinal(data.getBytes()));
        } catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException("Algorithm for HmacSHA1 not found", e);
        } catch (InvalidKeyException e) {
            throw new IllegalArgumentException("Invalid Key for initialization sha1 HMAC algorithm", e);
        }
        return sign;
    }

    String calculateCookieValue(String yandexUid, long bannerId, Instant now, Duration cookieTtl) {
        long timestamp = now.getEpochSecond();
        long expire = now.plus(cookieTtl).getEpochSecond();
        String sign = calculateSign(expire, timestamp, yandexUid, bannerId);
        return expire + "." + bannerId + "_" + timestamp + "_" + sign;
    }

    String calculateCookieSecretKey(Instant now) {
        long timestamp = now.getEpochSecond();
        long operatorUid = authenticationSource.getAuthentication().getOperator().getUid();
        long days = timestamp / 86400;
        return TuneSecretKey.generateSecretKey(days, operatorUid, null);
    }

    private String buildUrl(String host, String cookieValue, String retpath, Instant now) {
        Optional<YandexDomain> yandexDomain = HostUtils.getYandexDomain(host);
        String tldHost = yandexDomain.isPresent() ? yandexDomain.get().getYandexDomain() : DEFAULT_YANDEX_DOMAIN;
        String url = "https://" + tldHost + "/portal/set/any/";
        String secretKey = calculateCookieSecretKey(now);
        FastUrlBuilder builder = new FastUrlBuilder(url)
                .addParam("sk", secretKey)
                .addParam(COOKIE_NAME, cookieValue)
                .addParam("retpath", retpath);
        return builder.build();
    }

    private void checkArguments(long bid, Duration cookieTtl) {
        if (!isBannerBelongToClient(bid)) {
            throw new AccessDeniedException(String.format(
                    "banner (bannerId = %s) doesn't belong to operator", bid));
        }
        if (MAX_COOKIE_TTL.compareTo(cookieTtl) < 0) {
            throw new IllegalStateException("Cookie ttl is too long");
        }
    }

    String calculateSetCookieUrl(String host, String yandexUid, long bannerId, String retpath,
                                 Duration cookieTtl) {
        Instant now = clock.instant();
        String cookieValue = calculateCookieValue(yandexUid, bannerId, now, cookieTtl);
        return buildUrl(host, cookieValue, retpath, now);
    }


    public String getSetCookieUrl(String host, String yandexUid, long bid, String retpath, Duration cookieTtl) {
        checkArguments(bid, cookieTtl);
        Long bannerId = getBannerIdForAiming(bid);
        checkState(bannerId != null, "Banner with bid = {} not found", bid);
        checkState(bannerId != 0, "Banner system BannerID is 0 for bid = {}", bid);
        return calculateSetCookieUrl(host, yandexUid, bannerId, retpath, cookieTtl);

    }

    public String getResetCookieUrl(String host, String retpath) {
        Instant now = clock.instant();
        return buildUrl(host, EMPTY_VALUE, retpath, now);
    }
}
