package ru.yandex.direct.web.entity.creative.model;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.creative.model.AdditionalData;
import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.model.CreativeType;
import ru.yandex.direct.core.entity.creative.model.ModerationInfo;
import ru.yandex.direct.core.entity.creative.model.ModerationInfoAspect;
import ru.yandex.direct.core.entity.creative.model.ModerationInfoHtml;
import ru.yandex.direct.core.entity.creative.model.ModerationInfoImage;
import ru.yandex.direct.core.entity.creative.model.ModerationInfoSound;
import ru.yandex.direct.core.entity.creative.model.ModerationInfoText;
import ru.yandex.direct.core.entity.creative.model.ModerationInfoVideo;
import ru.yandex.direct.core.entity.creative.model.VideoFormat;

import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class WebCreativeConverter {
    public static WebVideoCreative toWebCreative(Creative creative) {
        WebVideoCreative webCreative = new WebVideoCreative()
                .withClientId(creative.getClientId())
                .withLivePreviewUrl(getLivePreviewUrl(creative))
                .withResourceUrl(creative.getPreviewUrl())
                .withName(creative.getName())
                .withType(toWebCreativeType(creative.getType()))
                .withWidth(creative.getWidth())
                .withHeight(creative.getHeight())
                .withDuration(creative.getDuration())
                .withModerationInfo(ifNotNull(creative.getModerationInfo(),
                        moderationInfo -> toWebModerationInfo(creative.getModerationInfo())))
                .withAdditionalData(ifNotNull(creative.getAdditionalData(), WebCreativeConverter::toAdditionalData))
                .withStockCreativeId(creative.getStockCreativeId())
                .withHasPackshot(creative.getHasPackshot());

        return webCreative;
    }

    private static String getLivePreviewUrl(Creative creative) {
        // livePreviewUrl для оверлейных креативов не нужен
        if (creative.getType() == CreativeType.CPM_OVERLAY) {
            return null;
        } else {
            return creative.getLivePreviewUrl();
        }
    }

    private static WebAdditionalData toAdditionalData(AdditionalData additionalData) {
        return new WebAdditionalData()
                .withDuration(additionalData.getDuration())
                .withFormats(ifNotNull(additionalData.getFormats(),
                        videoFormats -> filterAndMapList(videoFormats,
                                f -> f.getHeight() != null && f.getWidth() != null, WebCreativeConverter::toWebVideoFormat)));
    }

    private static WebVideoFormat toWebVideoFormat(VideoFormat f) {
        return new WebVideoFormat().withHeight(f.getHeight()).withWidth(f.getWidth()).withType(f.getType()).withUrl(f.getUrl());
    }

    private static WebCreativeType toWebCreativeType(CreativeType creativeType) {
        switch (creativeType) {
            case PERFORMANCE:
                return WebCreativeType.PERFORMANCE;
            case CANVAS:
                return WebCreativeType.CANVAS;
            case VIDEO_ADDITION_CREATIVE:
            case CPC_VIDEO_CREATIVE:
            case CPM_VIDEO_CREATIVE:
            case CPM_OUTDOOR_CREATIVE:
            case CPM_INDOOR_CREATIVE:
            case CPM_AUDIO_CREATIVE:
                return WebCreativeType.VIDEO_ADDITION;
            case CPM_OVERLAY:
                return WebCreativeType.OVERLAY;
            case HTML5_CREATIVE:
                return WebCreativeType.HTML5_CREATIVE;
            default:
                throw new IllegalStateException("No such value: " + creativeType);
        }
    }

    private static WebModerationInfo toWebModerationInfo(ModerationInfo moderationInfo) {
        return new WebModerationInfo()
                .withAspects(mapList(moderationInfo.getAspects(),
                        aspect -> ifNotNull(aspect, WebCreativeConverter::toWebModerationInfoAspect)))
                .withHtml(WebCreativeConverter.toWebModerationInfoHtml(moderationInfo.getHtml()))
                .withImages(mapList(moderationInfo.getImages(),
                        image -> ifNotNull(image, WebCreativeConverter::toWebModerationInfoImage)))
                .withSounds(mapList(moderationInfo.getSounds(),
                        sound -> ifNotNull(sound, WebCreativeConverter::toWebModerationInfoSound)))
                .withTexts(mapList(moderationInfo.getTexts(),
                        text -> ifNotNull(text, WebCreativeConverter::toWebModerationInfoText)))
                .withVideos(mapList(moderationInfo.getVideos(),
                        video -> ifNotNull(video, WebCreativeConverter::toWebModerationInfoVideo)));
    }

    private static WebModerationInfoAspect toWebModerationInfoAspect(ModerationInfoAspect moderationInfoAspect) {
        return new WebModerationInfoAspect()
                .withHeight(moderationInfoAspect.getHeight())
                .withWidth(moderationInfoAspect.getWidth());
    }

    private static WebModerationInfoHtml toWebModerationInfoHtml(ModerationInfoHtml moderationInfoHtml) {
        return new WebModerationInfoHtml().withUrl(moderationInfoHtml.getUrl());
    }

    private static WebModerationInfoImage toWebModerationInfoImage(ModerationInfoImage moderationInfoImage) {
        return new WebModerationInfoImage()
                .withUrl(moderationInfoImage.getUrl())
                .withAlt(moderationInfoImage.getAlt());
    }

    private static WebModerationInfoSound toWebModerationInfoSound(ModerationInfoSound moderationInfoSound) {
        return new WebModerationInfoSound()
                .withUrl(moderationInfoSound.getUrl())
                .withStockId(moderationInfoSound.getStockId());
    }

    private static WebModerationInfoText toWebModerationInfoText(ModerationInfoText moderationInfoText) {
        return new WebModerationInfoText()
                .withColor(moderationInfoText.getColor())
                .withText(moderationInfoText.getText())
                .withType(moderationInfoText.getType());
    }

    private static WebModerationInfoVideo toWebModerationInfoVideo(ModerationInfoVideo moderationInfoVideo) {
        return new WebModerationInfoVideo()
                .withStockId(moderationInfoVideo.getStockId())
                .withUrl(moderationInfoVideo.getUrl());
    }

}
