package ru.yandex.direct.web.entity.creative.service;

import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.creative.model.Creative;
import ru.yandex.direct.core.entity.creative.model.CreativeType;
import ru.yandex.direct.core.entity.creative.repository.CreativeConstants;
import ru.yandex.direct.core.entity.creative.service.CreativeService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;
import ru.yandex.direct.web.entity.creative.model.SearchVideoCreativesResponse;
import ru.yandex.direct.web.entity.creative.model.WebCreativeConverter;
import ru.yandex.direct.web.validation.kernel.ValidationResultConversionService;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class CreativeWebService {
    private static final Set<CreativeType> ALLOWED_CREATIVE_TYPES =
            ImmutableSet.copyOf(StreamEx.of(CreativeConstants.VIDEO_TYPES)
                    .append(CreativeType.CPM_AUDIO_CREATIVE)
                    .append(CreativeType.CPM_OVERLAY)
                    .toSet());

    private final CreativeService creativeService;
    private final DirectWebAuthenticationSource directWebAuthenticationSource;
    private final ValidationResultConversionService validationResultConversionService;
    private final CreativeWebValidationService creativeWebValidationService;

    @Autowired
    public CreativeWebService(CreativeService creativeService,
                              DirectWebAuthenticationSource directWebAuthenticationSource,
                              ValidationResultConversionService validationResultConversionService,
                              CreativeWebValidationService creativeWebValidationService) {
        this.creativeService = creativeService;
        this.directWebAuthenticationSource = directWebAuthenticationSource;
        this.validationResultConversionService = validationResultConversionService;
        this.creativeWebValidationService = creativeWebValidationService;
    }

    public WebResponse searchVideoCreatives(List<Long> creativeIds) {
        ValidationResult<List<Long>, Defect> validation =
                creativeWebValidationService.validateCreativeIds(creativeIds);

        if (validation.hasAnyErrors()) {
            return validationResultConversionService.buildValidationResponse(validation);
        }

        ClientId clientId = directWebAuthenticationSource.getAuthentication().getSubjectUser().getClientId();
        List<Creative> creatives = creativeService.get(clientId, creativeIds, ALLOWED_CREATIVE_TYPES);
        return new SearchVideoCreativesResponse().withResult(mapList(creatives, WebCreativeConverter::toWebCreative));
    }
}
