package ru.yandex.direct.web.entity.deal.controller;

import java.util.List;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.web.annotations.AllowedSubjectRoles;
import ru.yandex.direct.web.core.model.WebErrorResponse;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.security.captcha.DisableAutoCaptcha;
import ru.yandex.direct.web.entity.deal.model.DealsChangeStatusResponse;
import ru.yandex.direct.web.entity.deal.model.GetDealsDetailsResponse;
import ru.yandex.direct.web.entity.deal.model.GetDealsListResponse;
import ru.yandex.direct.web.entity.deal.model.UpdateDeal;
import ru.yandex.direct.web.entity.deal.model.UpdateDealResponse;
import ru.yandex.direct.web.entity.deal.service.DealWebService;

import static ru.yandex.direct.web.core.security.authentication.DirectCookieAuthProvider.PARAMETER_ULOGIN;

@Controller
@RequestMapping(value = "/deal", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Api(tags = "deal")
@AllowedSubjectRoles({RbacRole.AGENCY})
public class DealController {
    public static final String SORT_BY = "sort_by";
    public static final String SORT_ORDER = "sort_order";
    public static final String NO_STAT = "no_stat";

    private final DealWebService dealWebService;

    @Autowired
    public DealController(DealWebService dealWebService) {
        this.dealWebService = dealWebService;
    }

    @ApiOperation(
            value = "getDealsList",
            httpMethod = "GET",
            nickname = "getDealsList"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = GetDealsListResponse.class)
            }
    )
    @RequestMapping(path = "/list", method = RequestMethod.GET)
    @ResponseBody
    public WebResponse getDeals(
            @RequestParam(value = SORT_BY, required = false) String sortBy,
            @RequestParam(value = SORT_ORDER, required = false) String sortOrder,
            @RequestParam(value = NO_STAT, required = false) boolean noStat,
            @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin) {
        return dealWebService.getDealsList(sortBy, sortOrder, noStat);
    }

    @ApiOperation(
            value = "getDealsDetails",
            httpMethod = "POST",
            nickname = "getDealsDetails"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = GetDealsDetailsResponse.class)
            }
    )
    @RequestMapping(path = "/details", method = RequestMethod.POST)
    @ResponseBody
    public WebResponse getDealsDetails(@RequestBody List<Long> dealIds,
                                       @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN, required =
                                               false) String subjectLogin) {
        return dealWebService.getDealsDetails(dealIds);
    }

    @ApiOperation(
            value = "updateDeals",
            httpMethod = "POST",
            nickname = "updateDeals"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = UpdateDealResponse.class)
            }
    )
    @PreAuthorizeWrite
    @DisableAutoCaptcha
    @RequestMapping(path = "/update_deals",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @ResponseBody
    public WebResponse updateDeals(@RequestBody UpdateDeal request,
                                   @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN, required =
                                           false) String subjectLogin) {
        return dealWebService.updateDeal(request);
    }

    @ApiOperation(
            value = "activateDeals",
            httpMethod = "POST",
            nickname = "activateDeals"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = DealsChangeStatusResponse.class)
            }
    )
    @PreAuthorizeWrite
    @DisableAutoCaptcha
    @RequestMapping(path = "/activate",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @ResponseBody
    public WebResponse activateDeals(@RequestBody List<Long> dealIds,
                                     @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN, required =
                                             false) String subjectLogin) {
        return dealWebService.activateDeals(dealIds);
    }

    @ApiOperation(
            value = "completeDeals",
            httpMethod = "POST",
            nickname = "completeDeals"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = DealsChangeStatusResponse.class)
            }
    )
    @PreAuthorizeWrite
    @DisableAutoCaptcha
    @RequestMapping(path = "/complete",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @ResponseBody
    public WebResponse completeDeals(@RequestBody List<Long> dealIds,
                                     @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN, required =
                                             false) String subjectLogin) {
        return dealWebService.completeDeals(dealIds);
    }

    @ApiOperation(
            value = "archiveDeals",
            httpMethod = "POST",
            nickname = "archiveDeals"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = DealsChangeStatusResponse.class)
            }
    )
    @PreAuthorizeWrite
    @DisableAutoCaptcha
    @RequestMapping(path = "/archive",
            method = RequestMethod.POST,
            consumes = MediaType.APPLICATION_JSON_UTF8_VALUE)
    @ResponseBody
    public WebResponse archiveDeals(@RequestBody List<Long> dealIds,
                                    @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN, required =
                                            false) String subjectLogin) {
        return dealWebService.archiveDeals(dealIds);
    }
}
