package ru.yandex.direct.web.entity.deal.controller;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.RandomUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import ru.yandex.direct.common.spring.TestingComponent;
import ru.yandex.direct.core.entity.deal.container.CampaignDeal;
import ru.yandex.direct.core.entity.deal.container.UpdateDealContainer;
import ru.yandex.direct.core.entity.deal.model.CompleteReason;
import ru.yandex.direct.core.entity.deal.model.Deal;
import ru.yandex.direct.core.entity.deal.model.DealPlacement;
import ru.yandex.direct.core.entity.deal.model.DealType;
import ru.yandex.direct.core.entity.deal.model.StatusAdfox;
import ru.yandex.direct.core.entity.deal.repository.DealRepository;
import ru.yandex.direct.core.entity.deal.service.DealService;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.Percent;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.web.annotations.AllowedOperatorRoles;
import ru.yandex.direct.web.annotations.AllowedSubjectRoles;
import ru.yandex.direct.web.core.model.WebErrorResponse;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.model.WebSuccessResponse;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;
import ru.yandex.direct.web.entity.SuccessResponse;
import ru.yandex.direct.web.entity.deal.model.DealsChangeStatusResponse;
import ru.yandex.direct.web.entity.deal.model.GetDealsListResponse;
import ru.yandex.direct.web.validation.kernel.ValidationResultConversionService;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.dbschema.ppc.tables.Deals.DEALS;
import static ru.yandex.direct.dbschema.ppcdict.tables.DealsAdfox.DEALS_ADFOX;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.web.core.security.authentication.DirectCookieAuthProvider.PARAMETER_ULOGIN;

@RequestMapping(value = "/deal_test", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Api(tags = "deal_test")
@AllowedOperatorRoles({RbacRole.SUPER})
@AllowedSubjectRoles({RbacRole.AGENCY})
@TestingComponent
public class DealTestController {
    private final DealService dealService;
    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;
    private final DirectWebAuthenticationSource directWebAuthenticationSource;
    private final ValidationResultConversionService validationResultConversionService;
    private final DealRepository dealRepository;

    @Autowired
    public DealTestController(DealService dealService,
                              DslContextProvider dslContextProvider,
                              ShardHelper shardHelper,
                              DirectWebAuthenticationSource directWebAuthenticationSource,
                              ValidationResultConversionService validationResultConversionService,
                              DealRepository dealRepository) {
        this.dealService = dealService;
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
        this.directWebAuthenticationSource = directWebAuthenticationSource;
        this.validationResultConversionService = validationResultConversionService;
        this.dealRepository = dealRepository;
    }

    @ApiOperation(
            value = "addRandomDeals",
            httpMethod = "POST",
            nickname = "addRandomDeals"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = DealsResponse.class)
            }
    )
    @RequestMapping(path = "/add_random_list", method = RequestMethod.POST)
    @ResponseBody
    public WebResponse addRandomDeals(Long count,
                                      @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN, required =
                                              false) String subjectLogin) {
        ClientId clientId = directWebAuthenticationSource.getAuthentication().getSubjectUser().getClientId();
        List<Deal> deals = new ArrayList<>();
        for (int i = 0; i < count; i++) {
            deals.add(generateRandomDeal(clientId));
        }
        MassResult massResult = dealService.addDeals(clientId, deals);
        if (!massResult.isSuccessful()) {
            return new WebErrorResponse("Не удалось добавить сделки",
                    StreamEx.of(massResult.getValidationResult().flattenErrors()).joining("\\n"));
        }
        return new DealsResponse().withResult(deals);
    }

    @ApiOperation(
            value = "addRandomDeal",
            httpMethod = "POST",
            nickname = "addRandomDeal"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = DealResponse.class)
            }
    )
    @RequestMapping(path = "/add_random", method = RequestMethod.POST)
    @ResponseBody
    public WebResponse addRandomDeal(
            @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin) {
        ClientId clientId = directWebAuthenticationSource.getAuthentication().getSubjectUser().getClientId();
        Deal deal = generateRandomDeal(clientId);
        dealService.addDeals(clientId, singletonList(deal));
        return new DealResponse().withResult(deal);
    }

    @ApiOperation(
            value = "addOne",
            httpMethod = "POST",
            nickname = "addOne"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = GetDealsListResponse.class)
            }
    )
    @RequestMapping(path = "/add_one", method = RequestMethod.POST)
    @ResponseBody
    public WebResponse addOne(@RequestBody Deal deal,
                              @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin) {
        ClientId clientId = directWebAuthenticationSource.getAuthentication().getSubjectUser().getClientId();
        deal.withClientId(clientId.asLong());
        dealService.addDeals(clientId, singletonList(deal));
        return new DealResponse().withResult(deal);
    }

    private static final Long MAX = 10000000L;

    private Deal generateRandomDeal(ClientId clientId) {

        Long id = RandomUtils.nextLong(0, MAX);
        Deal deal = new Deal();
        deal.withId(id)
                .withDateCreated(LocalDateTime.now())
                .withPublisherName("Publisher name for " + id)
                .withAdfoxName("Adfox name for " + id)
                .withAdfoxDescription("Adfox description for " + id)
                .withAdfoxStatus(StatusAdfox.CREATED)
                .withDescription("Description for " + id)
                .withDateStart(LocalDateTime.now().plusDays(1))
                .withDateEnd(LocalDateTime.now().plusDays(15))
                .withTargetingsText("Targetings Text for " + id)
                .withContacts("Contacts for " + id)
                .withCpm(BigDecimal.valueOf(RandomUtils.nextLong(0, MAX)))
                .withExpectedImpressionsPerWeek(RandomUtils.nextLong(0, MAX))
                .withExpectedMoneyPerWeek(RandomUtils.nextLong(0, MAX))
                .withAgencyFeePercent(Percent.fromPercent(BigDecimal.valueOf(10L)))
                .withAgencyFeeType(12L)
                .withMarginRatio(Percent.fromPercent(BigDecimal.valueOf(5L)))
                .withDealType(DealType.PREFERRED_DEAL)
                .withDealJson("{}")
                .withCurrencyCode(CurrencyCode.RUB)
                .withClientId(clientId.asLong())
                .withPlacements(Collections.singletonList(new DealPlacement()
                        .withPageId(123L)
                        .withImpId(Arrays.asList(1L, 2L))));
        return deal;
    }

    @ApiOperation(
            value = "completeWithReason",
            httpMethod = "POST",
            nickname = "completeWithReason"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = DealsResponse.class)
            }
    )
    @RequestMapping(path = "/complete", method = RequestMethod.POST)
    @ResponseBody
    public WebResponse completeWithReason(@RequestBody List<Long> dealIds,
                                          @RequestParam(value = "complete_reason") CompleteReason completeReason,
                                          @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN,
                                                  required = false) String subjectLogin) {
        ClientId clientId = directWebAuthenticationSource.getAuthentication().getSubjectUser().getClientId();
        MassResult<Long> result = dealService.completeDeals(clientId, dealIds, completeReason, Applicability.FULL);
        if (result.getErrorCount() > 0) {
            return validationResultConversionService.buildValidationResponse(result);
        }
        return new DealsResponse().withResult(mapList(dealIds, id -> {
            Deal deal = new Deal();
            deal.withId(id);
            return deal;
        }));
    }

    @ApiOperation(
            value = "deleteDeals",
            httpMethod = "POST",
            nickname = "deleteDeals"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = DealsResponse.class)
            }
    )
    @RequestMapping(path = "/delete", method = RequestMethod.POST)
    @ResponseBody
    public WebResponse deleteDeals(
            @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin) {
        ClientId agencyId = directWebAuthenticationSource.getAuthentication().getSubjectUser().getClientId();
        long agencyUid = directWebAuthenticationSource.getAuthentication().getSubjectUser().getUid();
        return new DealsChangeStatusResponse().withResult(deleteDeals(agencyUid, agencyId));
    }

    @ApiOperation(
            value = "deleteDealsByIds",
            httpMethod = "POST",
            nickname = "deleteDealsByIds"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = DealsResponse.class)
            }
    )
    @RequestMapping(path = "/delete_by_ids", method = RequestMethod.POST)
    @ResponseBody
    public WebResponse deleteDealsByIds(@RequestBody List<Long> dealIds,
                                        @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN, required
                                                = false) String subjectLogin) {
        ClientId agencyId = directWebAuthenticationSource.getAuthentication().getSubjectUser().getClientId();
        long agencyUid = directWebAuthenticationSource.getAuthentication().getSubjectUser().getUid();
        return new DealsChangeStatusResponse().withResult(deleteDealsByIds(agencyUid, agencyId, dealIds));
    }

    @ApiOperation(
            value = "addCampaignLinks",
            httpMethod = "POST",
            nickname = "addCampaignLinks",
            notes = "Фейковая привязка кампаний к сделкам. Ничего не валидирует"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
                    @ApiResponse(code = 200, message = "Ok", response = WebResponse.class)
            }
    )
    @RequestMapping(path = "/add_links", method = RequestMethod.POST)
    @ResponseBody
    public WebResponse addCampaignLinks(
            @RequestBody List<CampaignDeal> links,
            @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin) {
        shardHelper.groupByShard(links, ShardKey.CID, CampaignDeal::getCampaignId)
                .chunkedByDefault()
                .stream()
                .forKeyValue(dealRepository::linkCampaigns);
        return new WebSuccessResponse();
    }

    public static class DealResponse extends SuccessResponse<Deal> {
    }

    public static class DealsResponse extends SuccessResponse<List<Deal>> {
    }

    //тестовое удаление сделок
    private List<Long> deleteDeals(long agencyUid, ClientId agencyId) {
        List<Deal> deals = dealService.getDealsBrief(agencyId);
        List<Long> dealIds = mapList(deals, Deal::getId);
        return deleteDealsByIds(agencyUid, agencyId, dealIds);
    }

    //тестовое удаление сделок DIRECT-95582
    private List<Long> deleteDealsByIds(long agencyUid, ClientId agencyId, List<Long> dealIds) {
        int shard = shardHelper.getShardByClientIdStrictly(agencyId);
        Map<Long, List<Long>> linkedCampaigns = dealService.getLinkedCampaignsByClientId(agencyId);
        List<CampaignDeal> campaignsDeals = new ArrayList<>();
        for (Long dealId : dealIds) {
            List<Long> camps = linkedCampaigns.getOrDefault(dealId, emptyList());
            if (!camps.isEmpty()) {
                for (Long campaignId : camps) {
                    campaignsDeals.add(new CampaignDeal().withDealId(dealId).withCampaignId(campaignId));
                }
            }
        }
        dealService
                .updateDeal(agencyUid, agencyId, new UpdateDealContainer().withRemoved(campaignsDeals));
        dslContextProvider.ppc(shard)
                .deleteFrom(DEALS)
                .where(DEALS.DEAL_ID.in(dealIds))
                .execute();
        dslContextProvider.ppcdict()
                .deleteFrom(DEALS_ADFOX)
                .where(DEALS_ADFOX.DEAL_ID.in(dealIds))
                .execute();
        return dealIds;
    }
}
