package ru.yandex.direct.web.entity.deal.model;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.deal.container.CampaignDeal;
import ru.yandex.direct.core.entity.deal.container.DealStat;
import ru.yandex.direct.core.entity.deal.container.UpdateDealContainer;
import ru.yandex.direct.core.entity.deal.model.CompleteReason;
import ru.yandex.direct.core.entity.deal.model.Deal;
import ru.yandex.direct.core.entity.deal.model.DealPlacement;
import ru.yandex.direct.core.entity.deal.model.DealType;
import ru.yandex.direct.core.entity.deal.model.StatusDirect;
import ru.yandex.direct.core.entity.deal.model.UpdatableDeal;
import ru.yandex.direct.core.entity.deal.service.DealService;
import ru.yandex.direct.core.entity.placements.model.Placement;

import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class WebDealConverter {
    public static void addStatToWebDeal(WebDeal webDeal, DealStat dealStat) {
        webDeal.withShows(dealStat.getShows())
                .withClicks(dealStat.getClicks())
                .withSpent(dealStat.getSpent())
                .withCpc(dealStat.getCpc())
                .withCpm(dealStat.getCpm())
                .withCtr(dealStat.getCtr() == null ? null : dealStat.getCtr().asPercent());
    }

    public static void addStatsToWebDeals(List<WebDeal> webDeals, Collection<DealStat> dealStat) {
        Map<Long, DealStat> dealsStatById = StreamEx.of(dealStat)
                .mapToEntry(DealStat::getId)
                .invert()
                .toMap();

        StreamEx.of(webDeals)
                .filter(deal -> dealsStatById.containsKey(deal.getId()))
                .forEach(deal -> WebDealConverter.addStatToWebDeal(deal, dealsStatById.get(deal.getId())));
    }

    public static UpdateDealContainer toUpdateDealContainer(UpdateDeal updateDeal) {
        return new UpdateDealContainer()
                .withAdded(mapList(updateDeal.getAddLinks(), WebDealConverter::toCampaignsDeal))
                .withRemoved(mapList(updateDeal.getRemoveLinks(), WebDealConverter::toCampaignsDeal))
                .withDeals(mapList(updateDeal.getUpdatableWebDealList(), WebDealConverter::toUpdatableDeal));
    }

    public static UpdateDeal toUpdateDeal(UpdateDealContainer updateDealContainer) {
        return new UpdateDeal()
                .withAddLinks(mapList(updateDealContainer.getAdded(), WebDealConverter::toWebDealIdWithCampaign))
                .withRemoveLinks(mapList(updateDealContainer.getRemoved(), WebDealConverter::toWebDealIdWithCampaign))
                .withUpdatableWebDealList(
                        mapList(updateDealContainer.getDeals(), WebDealConverter::toUpdatableWebDeal));
    }

    public static UpdatableDeal toUpdatableDeal(UpdatableWebDeal updatableWebDeal) {
        return new Deal()
                .withId(updatableWebDeal.getDealId())
                .withName(updatableWebDeal.getName())
                .withDescription(updatableWebDeal.getDescription());
    }

    public static UpdatableWebDeal toUpdatableWebDeal(UpdatableDeal updatableDeal) {
        return new UpdatableWebDeal()
                .withDealId(updatableDeal.getId())
                .withName(updatableDeal.getName())
                .withDescription(updatableDeal.getDescription());
    }

    private static CampaignDeal toCampaignsDeal(WebDealIdWithCampaign webModel) {
        return new CampaignDeal().withDealId(webModel.getDealId()).withCampaignId(webModel.getCampaignId());
    }

    private static WebDealIdWithCampaign toWebDealIdWithCampaign(CampaignDeal campaignDeal) {
        return new WebDealIdWithCampaign().withDealId(campaignDeal.getDealId())
                .withCampaignId(campaignDeal.getCampaignId());
    }

    public static WebDeal toWebDeal(Deal deal) {
        return new WebDeal()
                .withId(deal.getId())
                .withName(Optional.ofNullable(deal.getName()).orElse(deal.getAdfoxName()))
                .withDealType(fromDealType(deal.getDealType()))
                .withMinPrice(deal.getAgencyMinPrice().bigDecimalValue())
                .withStatus(fromDirectStatus(deal.getDirectStatus()))
                .withCompleteReason(fromCompleteReason(deal.getCompleteReason()))
                .withCurrencyCode(String.valueOf(deal.getCurrencyCode()))
                .withAgencyID(deal.getClientId())
                .withContacts(deal.getContacts())
                .withDateStart(deal.getDateStart())
                .withDateEnd(deal.getDateEnd())
                .withDescription(Optional.ofNullable(deal.getDescription()).orElse(deal.getAdfoxDescription()))
                .withPublisherName(deal.getPublisherName())
                .withExpectedImpressionsPerWeek(deal.getExpectedImpressionsPerWeek())
                .withExpectedMoneyPerWeek(deal.getExpectedMoneyPerWeek())
                .withTargetingsText(deal.getTargetingsText())
                .withAgencyFeePercent(
                        deal.getAgencyFeePercent() == null ? BigDecimal.ZERO : deal.getAgencyFeePercent().asPercent());
    }

    public static WebDeal toWebDeal(Deal deal, Map<Long, Placement> placementsMap) {
        return toWebDeal(deal)
                .withPlacements(filterList(
                        mapList(deal.getPlacements(), p -> fromDealPlacement(p, placementsMap)),
                        Objects::nonNull
                ));
    }

    private static WebDealType fromDealType(DealType dealType) {
        if (dealType == DealType.PREFERRED_DEAL) {
            return WebDealType.PREFERRED_DEAL;
        }
        if (dealType == DealType.PRIVATE_MARKETPLACE) {
            return WebDealType.PRIVATE_MARKETPLACE;
        }
        return null;
    }

    private static WebDealStatus fromDirectStatus(StatusDirect dealType) {
        return WebDealStatus.valueOf(dealType.name());
    }

    private static WebDealCompleteReason fromCompleteReason(@Nullable CompleteReason completeReason) {
        if (completeReason == null) {
            return null;
        }
        return WebDealCompleteReason.valueOf(completeReason.name());
    }

    private static WebDealPlacement fromDealPlacement(@Nullable DealPlacement dealPlacement,
                                                      Map<Long, Placement> placementsMap) {
        if (dealPlacement == null) {
            return null;
        }

        Placement placement = placementsMap.get(dealPlacement.getPageId());
        if (placement == null) {
            return null;
        }

        String domain = placement.getDomain();
        Set<String> formats = DealService.extractFormatsFromPlacement(placement, dealPlacement);

        return new WebDealPlacement()
                .withPageId(dealPlacement.getPageId())
                .withDomain(domain)
                .withImpId(dealPlacement.getImpId())
                .withFormats(formats);
    }
}
