package ru.yandex.direct.web.entity.excel.controller;

import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.github.shyiko.mysql.binlog.GtidSet;
import com.google.common.collect.ImmutableMap;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import org.apache.poi.ss.util.CellReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RequestPart;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.multipart.MultipartFile;

import ru.yandex.direct.core.security.DirectAuthentication;
import ru.yandex.direct.core.security.authorization.PreAuthorizeRead;
import ru.yandex.direct.core.security.authorization.PreAuthorizeWrite;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.model.UidAndClientId;
import ru.yandex.direct.excel.processing.exception.ExcelValidationException;
import ru.yandex.direct.excel.processing.model.internalad.ExcelImportResult;
import ru.yandex.direct.excelmapper.exceptions.CantReadEmptyException;
import ru.yandex.direct.excelmapper.exceptions.CantReadFormatException;
import ru.yandex.direct.excelmapper.exceptions.CantReadRangeTooSmallException;
import ru.yandex.direct.excelmapper.exceptions.CantReadUnexpectedDataException;
import ru.yandex.direct.excelmapper.exceptions.ExcelMapperReadException;
import ru.yandex.direct.excelmapper.exceptions.InvalidCellDataFormatException;
import ru.yandex.direct.grid.core.entity.sync.service.MysqlStateService;
import ru.yandex.direct.grid.processing.util.ResponseConverter;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.web.core.model.WebErrorResponse;
import ru.yandex.direct.web.core.model.WebResponse;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;
import ru.yandex.direct.web.entity.excel.model.ExcelImportResponse;
import ru.yandex.direct.web.entity.excel.model.ExcelImportResultInfo;
import ru.yandex.direct.web.entity.excel.model.ValidationResponseWithExcelFileUrl;
import ru.yandex.direct.web.entity.excel.model.internalad.InternalAdExportInfo;
import ru.yandex.direct.web.entity.excel.model.internalad.InternalAdExportRequest;
import ru.yandex.direct.web.entity.excel.model.internalad.InternalAdExportResponse;
import ru.yandex.direct.web.entity.excel.model.internalad.InternalAdImportInfo;
import ru.yandex.direct.web.entity.excel.model.internalad.InternalAdImportMode;
import ru.yandex.direct.web.entity.excel.model.internalad.InternalAdImportRequest;
import ru.yandex.direct.web.entity.excel.model.internalad.InternalAdTemplateExportInfo;
import ru.yandex.direct.web.entity.excel.model.internalad.InternalAdTemplateExportRequest;
import ru.yandex.direct.web.entity.excel.model.internalad.InternalAdTemplateExportResponse;
import ru.yandex.direct.web.entity.excel.model.internalad.InternalAdUploadExcelFileResponse;
import ru.yandex.direct.web.entity.excel.service.InternalAdExcelWebService;
import ru.yandex.direct.web.entity.excel.service.InternalAdValidationService;
import ru.yandex.direct.web.validation.model.ValidationResponse;
import ru.yandex.direct.web.validation.model.WebDefect;
import ru.yandex.direct.web.validation.model.WebValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.core.validation.ValidationUtils.hasValidationIssues;
import static ru.yandex.direct.excel.processing.model.internalad.ExcelFetchedData.ADS_PATH;
import static ru.yandex.direct.excel.processing.model.internalad.ExcelFetchedData.AD_GROUPS_PATH;
import static ru.yandex.direct.grid.processing.service.operator.OperatorAllowedActionsUtils.canImportExcelForInternalAds;
import static ru.yandex.direct.validation.result.PathHelper.field;
import static ru.yandex.direct.validation.result.PathHelper.path;
import static ru.yandex.direct.web.core.security.authentication.DirectCookieAuthProvider.PARAMETER_ULOGIN;

@Controller
@RequestMapping(value = "/excel",
        consumes = MediaType.APPLICATION_JSON_UTF8_VALUE,
        produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
@Api(tags = "excel")
public class ExcelController {

    private static final Logger logger = LoggerFactory.getLogger(ExcelController.class);

    static final Map<Class<? extends ExcelMapperReadException>, String> EXCEL_MAPPER_READ_EXCEPTION_TO_CODE =
            ImmutableMap.<Class<? extends ExcelMapperReadException>, String>builder()
                    .put(CantReadEmptyException.class, "CANT_READ_EMPTY")
                    .put(CantReadFormatException.class, "CANT_READ_FORMAT")
                    .put(CantReadRangeTooSmallException.class, "CANT_READ_RANGE_TOO_SMALL")
                    .put(CantReadUnexpectedDataException.class, "CANT_READ_UNEXPECTED_DATA")
                    .put(InvalidCellDataFormatException.class, "INVALID_CELL_DATA_FORMAT")
                    .build();

    public static final String EXCEL_FILE_PARAM = "excelFile";
    private static final String UNKNOWN_DEFECT_CODE = "unknown";
    static final String IMPORT_MODE_PARAM = "importMode";
    public static final String INTERNAL_AD_UPLOAD_EXCEL_FILE_FOR_IMPORT_AND_GET_DATA_CONTROLLER =
            "/internal_ad_upload_excel_file_for_import_and_get_data";

    private final DirectWebAuthenticationSource authenticationSource;
    private final InternalAdValidationService internalAdValidationService;
    private final InternalAdExcelWebService internalAdExcelWebService;
    private final MysqlStateService mysqlStateService;

    public ExcelController(DirectWebAuthenticationSource authenticationSource,
                           InternalAdValidationService internalAdValidationService,
                           InternalAdExcelWebService internalAdExcelWebService,
                           MysqlStateService mysqlStateService) {
        this.authenticationSource = authenticationSource;
        this.internalAdValidationService = internalAdValidationService;
        this.internalAdExcelWebService = internalAdExcelWebService;
        this.mysqlStateService = mysqlStateService;
    }

    @ApiOperation(
            value = "internalAdExport",
            httpMethod = "POST",
            nickname = "internalAdExport"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
            }
    )
    @PreAuthorizeRead
    @RequestMapping(path = "/internal_ad_export", method = RequestMethod.POST)
    @ResponseBody
    public WebResponse internalAdExport(
            @RequestBody InternalAdExportRequest request,
            @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin
    ) {
        DirectAuthentication auth = authenticationSource.getAuthentication();
        ClientId clientId = auth.getSubjectUser().getClientId();

        Result<String> exportResult = internalAdExcelWebService.exportInternal(clientId, request);
        if (hasValidationIssues(exportResult)) {
            return internalAdValidationService.buildValidationResponse(exportResult);
        }

        return new InternalAdExportResponse()
                .withResult(new InternalAdExportInfo()
                        .withExportFileUrl(exportResult.getResult())
                );
    }

    @ApiOperation(
            value = "internalAdUploadExcelFileForImportAndGetData",
            httpMethod = "POST",
            nickname = "internalAdUploadExcelFileForImportAndGetData"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
            }
    )
    @PreAuthorizeWrite
    @RequestMapping(path = INTERNAL_AD_UPLOAD_EXCEL_FILE_FOR_IMPORT_AND_GET_DATA_CONTROLLER,
            method = RequestMethod.POST, consumes = MediaType.MULTIPART_FORM_DATA_VALUE)
    @ResponseBody
    public WebResponse internalAdUploadExcelFileForImport(
            @RequestParam(IMPORT_MODE_PARAM) InternalAdImportMode importMode,
            @RequestPart(EXCEL_FILE_PARAM) MultipartFile excelFile,
            @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin) {
        DirectAuthentication auth = authenticationSource.getAuthentication();
        checkState(canImportExcelForInternalAds(auth.getOperator()),
                "internalAdUploadExcelFileForImportAndGetData not allowed for operator with role %s",
                auth.getOperator().getRole());
        ClientId clientId = auth.getSubjectUser().getClientId();

        Result<InternalAdImportInfo> result = internalAdExcelWebService
                .getDataFromExcelFileAndUploadFileToMds(clientId, importMode, excelFile);
        if (hasValidationIssues(result)) {
            return internalAdValidationService.buildValidationResponse(result);
        }

        return new InternalAdUploadExcelFileResponse()
                .withResult(result.getResult());
    }

    @ApiOperation(
            value = "internalAdImport",
            httpMethod = "POST",
            nickname = "internalAdImport"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
            }
    )
    @PreAuthorizeRead
    @RequestMapping(path = "/internal_ad_import", method = RequestMethod.POST)
    @ResponseBody
    public WebResponse internalAdImport(
            @RequestBody InternalAdImportRequest request,
            @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin
    ) {
        DirectAuthentication auth = authenticationSource.getAuthentication();
        checkState(canImportExcelForInternalAds(auth.getOperator()),
                "internal_ad_import not allowed for operator with role %s", auth.getOperator().getRole());

        Long operatorUid = auth.getOperator().getUid();
        UidAndClientId uidAndClientId = UidAndClientId.of(
                auth.getSubjectUser().getUid(), auth.getSubjectUser().getClientId());

        Result<ExcelImportResult> importResult =
                internalAdExcelWebService.importInternal(operatorUid, uidAndClientId, request);
        if (hasValidationIssues(importResult)) {
            ValidationResponse validationResponse = internalAdValidationService.buildValidationResponse(importResult);
            logger.warn("Got validationResponse: {}", JsonUtils.toJson(validationResponse));
            return validationResponse;
        }

        String excelFileUrl = importResult.getResult().getExcelFileUrl();
        if (importResult.getResult().hasValidationIssues()) {
            var validationResponse = getValidationResponseWithExcelFileUrl(importResult.getResult().getAdGroupsResult(),
                    importResult.getResult().getAdsResult(), excelFileUrl);
            logger.warn("Got validationResponseWithExcelFileUrl: {}", JsonUtils.toJson(validationResponse));
            return validationResponse;
        }

        if (request.getOnlyValidation()) {
            return toExcelImportResponseWithEmptyIds(excelFileUrl);
        }

        GtidSet.UUIDSet currentServerGtidSet = mysqlStateService.getCurrentServerGtidSet(uidAndClientId.getClientId());
        return toExcelImportResponse(importResult.getResult(), currentServerGtidSet.toString());
    }

    @ApiOperation(
            value = "internalAdTemplateExport",
            httpMethod = "POST",
            nickname = "internalAdTemplateExport"
    )
    @ApiResponses(
            {
                    @ApiResponse(code = 400, message = "Bad params", response = WebErrorResponse.class),
                    @ApiResponse(code = 403, message = "Permission denied", response = WebErrorResponse.class),
            }
    )
    @PreAuthorizeRead
    @RequestMapping(path = "/internal_ad_template_export", method = RequestMethod.POST)
    @ResponseBody
    public WebResponse internalAdTemplateExport(
            @RequestBody InternalAdTemplateExportRequest request,
            @SuppressWarnings("unused") @RequestParam(value = PARAMETER_ULOGIN, required = false) String subjectLogin
    ) {
        DirectAuthentication auth = authenticationSource.getAuthentication();
        ClientId clientId = auth.getSubjectUser().getClientId();

        Result<String> exportResult = internalAdExcelWebService.exportInternalTemplate(clientId, request);
        if (hasValidationIssues(exportResult)) {
            return internalAdValidationService.buildValidationResponse(exportResult);
        }

        return new InternalAdTemplateExportResponse()
                .withResult(new InternalAdTemplateExportInfo()
                        .withExportFileUrl(exportResult.getResult())
                );
    }

    private ValidationResponseWithExcelFileUrl getValidationResponseWithExcelFileUrl(MassResult<Long> adGroupsResult,
                                                                                     MassResult<Long> adsResult,
                                                                                     String url) {
        WebValidationResult adGroupsValidationResult = internalAdValidationService
                .buildWebValidationResult(adGroupsResult.getValidationResult(), path(field(AD_GROUPS_PATH)));
        WebValidationResult adsValidationResult = internalAdValidationService
                .buildWebValidationResult(adsResult.getValidationResult(), path(field(ADS_PATH)));

        //noinspection UnnecessaryLocalVariable
        var mergeResult = adGroupsValidationResult;
        mergeResult.addErrors(adsValidationResult.getErrors());
        mergeResult.addWarnings(adsValidationResult.getWarnings());

        return new ValidationResponseWithExcelFileUrl(mergeResult, url);
    }

    private static WebResponse toExcelImportResponseWithEmptyIds(String excelFileUrl) {
        return new ExcelImportResponse()
                .withResult(new ExcelImportResultInfo()
                        .withAddedOrUpdatedAdGroupIds(Collections.emptyList())
                        .withAddedOrUpdatedAdIds(Collections.emptyList())
                        .withExcelFileUrl(excelFileUrl)
                );
    }

    private static WebResponse toExcelImportResponse(ExcelImportResult result, String mutationId) {
        var importResultInfo = new ExcelImportResultInfo()
                .withAddedOrUpdatedAdGroupIds(
                        ResponseConverter.getSuccessfullyResults(result.getAdGroupsResult(), String::valueOf))
                .withAddedOrUpdatedAdIds(
                        ResponseConverter.getSuccessfullyResults(result.getAdsResult(), String::valueOf))
                .withMutationId(mutationId)
                .withExcelFileUrl(result.getExcelFileUrl());

        return new ExcelImportResponse()
                .withResult(importResultInfo);
    }

    @ExceptionHandler(ExcelMapperReadException.class)
    @ResponseBody
    public ValidationResponse excelMapperReadExceptionHandler(ExcelMapperReadException exception) {
        var webDefect = new WebDefect()
                .withCode(EXCEL_MAPPER_READ_EXCEPTION_TO_CODE.getOrDefault(exception.getClass(), UNKNOWN_DEFECT_CODE))
                .withPath(CellReference.convertNumToColString(exception.getColumnIndex()) + (exception.getRowIndex() + 1))
                .withParams(exception.getColumns());
        logger.warn("Got ExcelMapperReadException: {}", JsonUtils.toJson(webDefect));

        var result = new WebValidationResult()
                .addErrors(List.of(webDefect));
        return new ValidationResponse(result);
    }

    @ExceptionHandler({ExcelValidationException.class})
    @ResponseBody
    public ValidationResponse inputDataError(ExcelValidationException e) {
        var webDefect = new WebDefect()
                .withCode(e.getDefectId().getCode())
                .withPath(e.getPath().toString())
                .withParams(e.getParams());
        logger.warn("Got ExportExcelValidationException: {}", JsonUtils.toJson(webDefect));

        var result = new WebValidationResult()
                .addErrors(List.of(webDefect));
        return new ValidationResponse(result);
    }

}
