package ru.yandex.direct.web.entity.excel.service;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.regex.Pattern;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.poi.ss.usermodel.Workbook;

import ru.yandex.direct.core.entity.adgroup.model.InternalAdGroup;
import ru.yandex.direct.core.entity.banner.model.InternalBanner;
import ru.yandex.direct.core.entity.banner.model.TemplateVariable;
import ru.yandex.direct.core.entity.image.model.BannerImageFormat;
import ru.yandex.direct.core.entity.image.model.BannerImageFromPool;
import ru.yandex.direct.core.entity.internalads.model.ReadOnlyDirectTemplateResource;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.excel.processing.model.ObjectType;
import ru.yandex.direct.excel.processing.model.internalad.AdGroupAdditionalTargetingRepresentation;
import ru.yandex.direct.excel.processing.model.internalad.ExcelSheetFetchedData;
import ru.yandex.direct.excel.processing.model.internalad.InternalAdExportParameters;
import ru.yandex.direct.excel.processing.model.internalad.InternalAdGroupRepresentation;
import ru.yandex.direct.excel.processing.model.internalad.InternalBannerRepresentation;
import ru.yandex.direct.excel.processing.model.internalad.mappers.AdGroupAdditionalTargetingMapperSettings;
import ru.yandex.direct.excel.processing.utils.ExcelMapperUtils;
import ru.yandex.direct.web.entity.excel.model.ExcelFileKey;
import ru.yandex.direct.web.entity.excel.model.internalad.AdGroupAdditionalTargetingInfo;
import ru.yandex.direct.web.entity.excel.model.internalad.AdGroupInfo;
import ru.yandex.direct.web.entity.excel.model.internalad.AdInfo;
import ru.yandex.direct.web.entity.excel.model.internalad.ImageInfo;
import ru.yandex.direct.web.entity.excel.model.internalad.InternalAdExportMode;
import ru.yandex.direct.web.entity.excel.model.internalad.InternalAdExportRequest;
import ru.yandex.direct.web.entity.excel.model.internalad.InternalAdImportMode;
import ru.yandex.direct.web.entity.excel.model.internalad.TemplateVariableInfo;
import ru.yandex.inside.mds.MdsFileKey;

import static java.util.Map.entry;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
class ExcelWebConverter {

    private static final Pattern LIKE_IMAGE_HASH_PATTERN = Pattern.compile("[A-Za-z0-9_\\-]*");
    private static final Map<InternalAdImportMode, Set<ObjectType>> IMPORT_MODE_TO_OBJECT_TYPES =
            Map.ofEntries(
                    entry(InternalAdImportMode.ONLY_AD_GROUPS, Set.of(ObjectType.AD_GROUP)),
                    entry(InternalAdImportMode.ONLY_ADS, Set.of(ObjectType.AD)),
                    entry(InternalAdImportMode.AD_GROUPS_WITH_ADS, Set.of(ObjectType.AD_GROUP, ObjectType.AD))
            );

    private ExcelWebConverter() {
    }

    static Set<ObjectType> importModeToObjectTypes(InternalAdImportMode importMode) {
        if (IMPORT_MODE_TO_OBJECT_TYPES.containsKey(importMode)) {
            return IMPORT_MODE_TO_OBJECT_TYPES.get(importMode);
        }

        throw new IllegalStateException("unknown import mode: " + importMode);
    }

    static InternalAdExportParameters requestToInternalAdExportParameters(ClientId clientId,
                                                                          InternalAdExportRequest request) {
        return new InternalAdExportParameters()
                .withClientId(clientId)
                .withExportAdGroupsWithAds(request.getExportMode() == InternalAdExportMode.AD_GROUPS_WITH_ADS)
                .withCampaignIds(request.getCampaignIds())
                .withAdGroupIds(request.getAdGroupIds())
                .withAdIds(request.getAdIds())
                .withHideEmptyColumns(request.getHideEmptyColumns());
    }

    static byte[] toBytes(Workbook workbook) {
        try (ByteArrayOutputStream os = new ByteArrayOutputStream()) {
            workbook.write(os);
            return os.toByteArray();
        } catch (IOException e) {
            throw new RuntimeException("Couldn't get bytes from workbook");
        }
    }

    static List<AdInfo> toAdsInfo(List<ExcelSheetFetchedData<InternalBannerRepresentation>> adsSheets,
                                  List<ReadOnlyDirectTemplateResource> templateResources,
                                  Map<String, BannerImageFormat> bannerImageFormatsByHash,
                                  Map<String, BannerImageFromPool> bannerImageFromPoolsByHashes,
                                  Set<Long> imageTemplateResourceIds) {
        List<InternalBannerRepresentation> adRepresentations = flatMap(adsSheets, ExcelSheetFetchedData::getObjects);
        Map<String, String> imageFileNameByHashes = EntryStream.of(bannerImageFromPoolsByHashes)
                .mapValues(BannerImageFromPool::getName)
                .toImmutableMap();
        Map<Long, String> resourceDescriptionsById = listToMap(templateResources,
                ReadOnlyDirectTemplateResource::getId, ReadOnlyDirectTemplateResource::getDescription);

        return mapList(adRepresentations, representation -> new AdInfo()
                .withAdGroupId(ifNotNull(representation.getAdGroupId(), String::valueOf))
                .withAdGroupName(representation.getAdGroupName())
                .withId(ifNotNull(representation.getBanner().getId(), String::valueOf))
                .withDescription(representation.getBanner().getDescription())
                .withTemplateVariables(getTemplateVariableInfos(representation.getBanner(), resourceDescriptionsById))
                .withImages(getImagesInfo(representation.getBanner().getTemplateVariables(),
                        bannerImageFormatsByHash, imageFileNameByHashes, imageTemplateResourceIds)));
    }

    private static List<TemplateVariableInfo> getTemplateVariableInfos(InternalBanner internalBanner,
                                                                       Map<Long, String> resourceDescriptionsById) {
        return StreamEx.of(internalBanner.getTemplateVariables())
                .mapToEntry(TemplateVariable::getTemplateResourceId, TemplateVariable::getInternalValue)
                .mapKeys(resourceDescriptionsById::get)
                .mapKeyValue((description, value) -> new TemplateVariableInfo()
                        .withDescription(description)
                        .withValue(value))
                .toImmutableList();
    }

    /**
     * Возвращает параметры изображений для картиночных переменных
     * Если параметров изображения нет в {@param bannerImageFormatsByHash}, то возвращает флаг needUpload = true
     *
     * @param templateVariables          переменные шаблона, среди которых могут быть картиночные переменные с ранее
     *                                   загруженными изображениями или которые надо загрузить
     * @param existingImageFormatsByHash параметры ранее загруженных изображений
     * @param imageFileNameByHashes      имя файла изображения
     * @param imageTemplateResourceIds   идентификаторы картиночных переменных
     */
    static List<ImageInfo> getImagesInfo(List<TemplateVariable> templateVariables,
                                         Map<String, BannerImageFormat> existingImageFormatsByHash,
                                         Map<String, String> imageFileNameByHashes,
                                         Set<Long> imageTemplateResourceIds) {
        return StreamEx.of(templateVariables)
                .filter(variable -> variable.getInternalValue() != null)
                .filter(variable -> imageTemplateResourceIds.contains(variable.getTemplateResourceId()))
                .map(TemplateVariable::getInternalValue)
                .map(imageHashOrInfo -> toImageInfo(imageHashOrInfo, existingImageFormatsByHash, imageFileNameByHashes))
                .toImmutableList();
    }

    private static ImageInfo toImageInfo(String imageHashOrInfo,
                                         Map<String, BannerImageFormat> existingImageFormatsByHash,
                                         Map<String, String> imageFileNameByHashes) {
        if (existingImageFormatsByHash.containsKey(imageHashOrInfo)) {
            return ImageInfo.fromBannerImage(imageFileNameByHashes.get(imageHashOrInfo),
                    existingImageFormatsByHash.get(imageHashOrInfo));
        }

        return toImageInfoNeedToUpload(imageHashOrInfo);
    }

    private static ImageInfo toImageInfoNeedToUpload(String imageFileNameOrUrl) {
        if (ExcelMapperUtils.isStringLikeUrl(imageFileNameOrUrl)) {
            return ImageInfo.fromUrl(imageFileNameOrUrl);
        }

        return ImageInfo.fromFileName(imageFileNameOrUrl);
    }

    static List<AdGroupInfo> getAdGroupsInfo(ExcelSheetFetchedData<InternalAdGroupRepresentation> adGroupsSheet) {
        return mapList(adGroupsSheet.getObjects(), adGroupRepresentation -> {
            InternalAdGroup adGroup = adGroupRepresentation.getAdGroup();
            return new AdGroupInfo()
                    .withId(ifNotNull(adGroup.getId(), String::valueOf))
                    .withName(adGroup.getName())
                    .withLevel(ifNotNull(adGroup.getLevel(), String::valueOf))
                    .withStartTime(adGroup.getStartTime())
                    .withFinishTime(adGroup.getFinishTime())
                    .withRf(adGroup.getRf())
                    .withRfReset(adGroup.getRfReset())
                    .withMaxClicksCount(adGroup.getMaxClicksCount())
                    .withMaxClicksPeriod(adGroup.getMaxClicksPeriod())
                    .withMaxStopsCount(adGroup.getMaxStopsCount())
                    .withMaxStopsPeriod(adGroup.getMaxStopsPeriod())
                    .withGeoIncluded(adGroupRepresentation.getGeoIncluded())
                    .withGeoExcluded(adGroupRepresentation.getGeoExcluded())
                    .withAdditionalTargetings(
                            toAdditionalTargetingInfos(adGroupRepresentation.getTargetingRepresentation()))
                    .withRetargetingCondition(adGroupRepresentation.getRetargetingConditionRepresentation().getRetargetingCondition());
        });
    }

    private static List<AdGroupAdditionalTargetingInfo> toAdditionalTargetingInfos(
            AdGroupAdditionalTargetingRepresentation representation) {
        return mapList(representation.getTargetingList(), targeting ->
                new AdGroupAdditionalTargetingInfo()
                        .withDescription(AdGroupAdditionalTargetingMapperSettings.getTargetingTitle(targeting))
                        .withValue(representation.getTargetingOriginalValueFromExcel(targeting))
        );
    }

    static MdsFileKey toMdsFileKey(ExcelFileKey excelFileKey) {
        int mdsGroupId = Integer.parseInt(excelFileKey.getMdsGroupId());
        return new MdsFileKey(mdsGroupId, excelFileKey.getFileName());
    }

    /**
     * Возвращает множество хэшей для картиночных переменных шаблона
     * Значение переменных фильтрируются по {@link ExcelWebConverter#LIKE_IMAGE_HASH_PATTERN},
     * т.к. в значениях можем получить имена файлов с кириллицей из-за чего запрос в базу упадет из-за кодировки...
     *
     * @param imageTemplateVariables картиночные переменные шаблона,
     *                               ожидается что в метод будут переданы только картиночные переменные шаблона
     */
    static Set<String> getTemplateVariableImageHashes(List<TemplateVariable> imageTemplateVariables) {
        return StreamEx.of(imageTemplateVariables)
                .map(TemplateVariable::getInternalValue)
                .filter(Objects::nonNull)
                .filter(LIKE_IMAGE_HASH_PATTERN.asMatchPredicate())
                .toImmutableSet();
    }

    static List<InternalAdGroup> getAdGroupsForAdd(ExcelSheetFetchedData<InternalAdGroupRepresentation> adGroupsSheet) {
        return StreamEx.of(adGroupsSheet.getObjects())
                .map(InternalAdGroupRepresentation::getAdGroup)
                .filter(adGroup -> adGroup.getId() == null)
                .toImmutableList();
    }

}
